/*
 * RWGSEquilibratorView.java
 */

package rwgsequilibrator;

import chem.ChemConstants;
import chem.Shomate;
import numeth.NMResult;
import org.jdesktop.application.Action;
import org.jdesktop.application.ResourceMap;
import org.jdesktop.application.SingleFrameApplication;
import org.jdesktop.application.FrameView;
import org.jdesktop.application.Task;
import org.jdesktop.application.TaskMonitor;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.text.DecimalFormat;
import javax.swing.Timer;
import javax.swing.Icon;
import javax.swing.JDialog;
import javax.swing.JFrame;
import numeth.NMJacEstEqns;
import numeth.NMJacEstimator;
import numeth.NMNewtRaphEqns;
import numeth.NMNewtRaphSolver;

/**
 * The application's main frame.
 */
public class RWGSEquilibratorView extends FrameView {

    public RWGSEquilibratorView(SingleFrameApplication app) {
        super(app);

        initComponents();

        // status bar initialization - message timeout, idle icon and busy animation, etc
        ResourceMap resourceMap = getResourceMap();
        int messageTimeout = resourceMap.getInteger("StatusBar.messageTimeout");
        messageTimer = new Timer(messageTimeout, new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                statusMessageLabel.setText("");
            }
        });
        messageTimer.setRepeats(false);
        int busyAnimationRate = resourceMap.getInteger("StatusBar.busyAnimationRate");
        for (int i = 0; i < busyIcons.length; i++) {
            busyIcons[i] = resourceMap.getIcon("StatusBar.busyIcons[" + i + "]");
        }
        busyIconTimer = new Timer(busyAnimationRate, new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                busyIconIndex = (busyIconIndex + 1) % busyIcons.length;
                statusAnimationLabel.setIcon(busyIcons[busyIconIndex]);
            }
        });
        idleIcon = resourceMap.getIcon("StatusBar.idleIcon");
        statusAnimationLabel.setIcon(idleIcon);
        progressBar.setVisible(false);

        // connecting action tasks to status bar via TaskMonitor
        TaskMonitor taskMonitor = new TaskMonitor(getApplication().getContext());
        taskMonitor.addPropertyChangeListener(new java.beans.PropertyChangeListener() {
            public void propertyChange(java.beans.PropertyChangeEvent evt) {
                String propertyName = evt.getPropertyName();
                if ("started".equals(propertyName)) {
                    if (!busyIconTimer.isRunning()) {
                        statusAnimationLabel.setIcon(busyIcons[0]);
                        busyIconIndex = 0;
                        busyIconTimer.start();
                    }
                    progressBar.setVisible(true);
                    progressBar.setIndeterminate(true);
                } else if ("done".equals(propertyName)) {
                    busyIconTimer.stop();
                    statusAnimationLabel.setIcon(idleIcon);
                    progressBar.setVisible(false);
                    progressBar.setValue(0);
                } else if ("message".equals(propertyName)) {
                    String text = (String)(evt.getNewValue());
                    statusMessageLabel.setText((text == null) ? "" : text);
                    messageTimer.restart();
                } else if ("progress".equals(propertyName)) {
                    int value = (Integer)(evt.getNewValue());
                    progressBar.setVisible(true);
                    progressBar.setIndeterminate(false);
                    progressBar.setValue(value);
                }
            }
        });

        // set default values for the initial conditions
        T = 400.;
        P = 1.0;
        ratio = 1.0;
        temperatureTextField.setValue(T);
        pressureTextField.setValue(P);
        feedRatioTextField.setValue(ratio);
        // instantiate a Shomate object for each species
        String[] CO2Params = {
            "298	1200	24.997350	55.186960	-33.691370	7.948387	-0.136638	-403.607500	228.243100	-393.522400",
            "1200	6000	58.166390	2.720074	-0.492289	0.038844	-6.447293	-425.918600	263.612500	-393.522400"
        };
        CO2 = new Shomate(CO2Params);
        String[] H2Params = {
            "298	1500	33.107800	-11.508000	11.609300	-2.844400	-0.159665	-9.991971	172.788000	0.000000",
            "1500	6000	34.143400	0.503927	0.372036	-0.038599	-8.074761	-21.218800	162.093000	0.000000"
        };
        H2 = new Shomate(H2Params);
        String[] COParams = {
            "298	1300	25.567590	6.096130	4.054656	-2.671301	0.131021	-118.008900	227.366500	-110.527100",
            "1300	6000	35.150700	1.300095	-0.205921	0.013550	-3.282780	-127.837500	231.712000	-110.527100"
        };
        CO = new Shomate(COParams);
        String[] H2OParams = {
            "500	1700	30.092000	6.832514	6.793435	-2.534480	0.082139	-250.881000	223.396700	-241.826400",
            "1700	6000	41.964260	8.622053	-1.499780	0.098119	-11.157640	-272.179700	219.780900	-241.826400"
        };
        H2O = new Shomate(H2OParams);
    }

    // Action for the Help menu About item
    @Action
    public void showAboutBox() {
        if (aboutBox == null) {
            JFrame mainFrame = RWGSEquilibratorApp.getApplication().getMainFrame();
            aboutBox = new RWGSEquilibratorAboutBox(mainFrame);
            aboutBox.setLocationRelativeTo(mainFrame);
        }
        RWGSEquilibratorApp.getApplication().show(aboutBox);
    }

    // Action for the Help menu User Guide item
    @Action
    public void showUserGuide() {

        if (userGuide == null) {
            JFrame mainFrame = RWGSEquilibratorApp.getApplication().getMainFrame();
            userGuide = new RWGSEquilibratorUserGuide(mainFrame);
            userGuide.setLocationRelativeTo(mainFrame);
        }
        RWGSEquilibratorApp.getApplication().show(userGuide);
    }

    // Action for the Equilibrate button
    @Action
    public Task equilibrate() {
        return new EquilibrateTask(getApplication());
    }

    private class EquilibrateTask extends org.jdesktop.application.Task<Object, Void>
        implements NMNewtRaphEqns, NMJacEstEqns {
        double temperature, pressure, feedRatio, conversion;
        double sRxn, hRxn, gRxn, K;
        double[] x, y;
        EquilibrateTask(org.jdesktop.application.Application app) {
            // Runs on the EDT.  Copy GUI state that
            // doInBackground() depends on from parameters
            // to EquilibrateTask fields, here.
            super(app);

            // If this is the first click
                // see if they want to use a log file and if so, open it.

            // Get the initial conditions from the input textfields
            temperature = ((Number)temperatureTextField.getValue()).doubleValue();
            pressure = ((Number)pressureTextField.getValue()).doubleValue();
            feedRatio = ((Number)feedRatioTextField.getValue()).doubleValue();

            equilibrateButton.setEnabled(false);
        }
        @Override protected Object doInBackground() {
            // Your Task's code here.  This method runs
            // on a background thread, so don't reference
            // the Swing GUI from here.
            String txt = "Calculating Species Thermo";
            setMessage(txt);
            int progress = 0;
            setProgress(0);
            
            // Calculate initial moles and volume
            n0H2 = 1.0;
            n0CO2 = feedRatio;

            // Calculate the thermo properties of each species
            double sCO2 = CO2.sStd(temperature).getDoubleValue();
            double sH2 = H2.sStd(temperature).getDoubleValue();
            double sCO = CO.sStd(temperature).getDoubleValue();
            double sH2O = H2O.sStd(temperature).getDoubleValue();
            double hCO2 = CO2.hForm(temperature).getDoubleValue();
            double hH2 = H2.hForm(temperature).getDoubleValue();
            double hCO = CO.hForm(temperature).getDoubleValue();
            double hH2O = H2O.hForm(temperature).getDoubleValue();
            double gCO2 = CO2.gForm(temperature).getDoubleValue();
            double gH2 = H2.gForm(temperature).getDoubleValue();
            double gCO = CO.gForm(temperature).getDoubleValue();
            double gH2O = H2O.gForm(temperature).getDoubleValue();
            txt = "Calculating Reaction Thermo";
            setMessage(txt);
            progress = 25;
            setProgress(progress);

            // Calculate the thermo properties for the reaction
            sRxn = sCO + sH2O - sCO2 - sH2;
            hRxn = hCO + hH2O - hCO2 - hH2;
            gRxn = gCO + gH2O - gCO2 - gH2;
            K = Math.exp(-gRxn*1000.0/ChemConstants.RgasE/temperature);
            txt = "Calculating Equilibrium Composition";
            setMessage(txt);
            progress = 50;
            setProgress(progress);

            // Calculate the equilibrium composition
            x = new double[4];
            y = new double[4];
            x[0] = n0CO2;
            x[1] = n0H2;
            x[2] = 0.0;
            x[3] = 0.0;
            progress = 75;
            setProgress(progress);
            NMNewtRaphSolver theSolver = new NMNewtRaphSolver(this);
            theSolver.setTol(1.0e-4);
            theSolver.setTreatAsNonNegative(true);
            NMResult theResult = theSolver.solve(x, y);
            percentCO2 = 100*x[0]/(n0CO2 + n0H2);
            percentH2 = 100*x[1]/(n0CO2 + n0H2);
            percentCO = 100*x[2]/(n0CO2 + n0H2);
            percentH2O = 100*x[3]/(n0CO2 + n0H2);

            // Calculate the conversion
            if (feedRatio <= 1) {
              // CO2 is the limiting reagent
              conversion = 100*(n0CO2 - x[0])/n0CO2;
            } else {
              // H2 is the limiting reagent
              conversion = 100*(n0H2 -x[1])/n0H2;
            }
            txt = "";
            setMessage(txt);
            progress = 100;
            setProgress(progress);
            return null;  // return your result
        }
        @Override protected void succeeded(Object result) {
            // Runs on the EDT.  Update the GUI based on
            // the result computed by doInBackground().

            // Write the results to the screen
            DecimalFormat noDecimal = new DecimalFormat("####");
            DecimalFormat threeSigFigs = new DecimalFormat("0.00E00");
            DecimalFormat notScientific = new DecimalFormat("##0.00");
            String txt = "<html>&#916;S(" + noDecimal.format(temperature);
            txt += ") = ";
            if (Math.abs(sRxn) < 0.1) {
                txt += threeSigFigs.format(sRxn) + " J/(mol K)";
            } else {
                txt += notScientific.format(sRxn) + " J/(mol K)";
            }
            entropyLabel.setText(txt);
            txt = "<html>&#916;H(" + noDecimal.format(temperature);
            txt += ") = ";
            if (Math.abs(hRxn) < 0.1) {
                txt += threeSigFigs.format(hRxn) + " kJ/mol";
            } else {
                txt += notScientific.format(hRxn) + " kJ/mol";
            }
            enthalpyLabel.setText(txt);
            txt = "<html>&#916;G(" + noDecimal.format(temperature);
            txt += ") = ";
            if (Math.abs(gRxn) < 0.1) {
                txt += threeSigFigs.format(gRxn) + " kJ/mol";
            } else {
                txt += notScientific.format(gRxn) + " kJ/mol";
            }
            freeEnergyLabel.setText(txt);
            txt = "<html>K(" + noDecimal.format(temperature);
            txt += ") = " + threeSigFigs.format(K);
            equilConstantLabel.setText(txt);
            if (percentCO2 <0.1) {
                txt = threeSigFigs.format(percentCO2);
            } else {
                txt = notScientific.format(percentCO2);
            }
            txt += " %";
            percentCO2Label.setText(txt);
            if (percentH2 <0.1) {
                txt = threeSigFigs.format(percentH2); 
            } else {
                txt = notScientific.format(percentH2);
            }
            txt += " %";
            percentH2Label.setText(txt);
            if (percentCO <0.1) {
                txt = threeSigFigs.format(percentCO); 
            } else {
                txt = notScientific.format(percentCO);
            }
            txt += " %";
            percentCOLabel.setText(txt);
            if (percentH2O <0.1) {
                txt = threeSigFigs.format(percentH2O); 
            } else {
                txt = notScientific.format(percentH2O);
            }
            txt += " %";
            percentH2OLabel.setText(txt);

            equilibrateButton.setEnabled(true);

            // set the conversion label

            txt= "<html>Limiting Reagent ";
            if (feedRatio < 1.0) {
                txt += "(CO<sub>2</sub>) ";
            }
            if (feedRatio > 1.0) {
                txt += "(H<sub>2</sub>) ";
            }
            txt += "Conversion is ";
            if (conversion <0.1) {
                txt += threeSigFigs.format(conversion);
            } else {
                txt += notScientific.format(conversion);
            }
            txt += " %";
            conversionLabel.setText(txt);

            equilibrateButton.setEnabled(true);

            // Write the results to the log file
        }

        public NMResult evalNewtRaphEqns(double[] x, double[] y) {
            NMResult theResult = new NMResult();
            // 0 = CO2, 1 = H2, 2 = CO, 3 = H2O
            y[0] = K*x[0]*x[1] - x[2]*x[3];
            y[1] = n0CO2 - x[2]- x[0];
            y[2] = n0H2 - x[1]- x[3];
            y[3] = 2*n0CO2 - 2*x[0] - x[2] - x[3];
            return theResult;
        }

        public NMResult evalNewtRaphJac(double[] x, double[][] jac) {
            NMJacEstimator jacEstimator = new NMJacEstimator(this);
            NMResult theResult = new NMResult();
            jacEstimator.estimate(4, x, jac, 1.0E-4);
            return theResult;
        }

        public NMResult evalJacEstEqns(double[] x, double[] y) {
            NMResult theResult = evalNewtRaphEqns(x,y);
            return theResult;
        }
    }

    public void clearResults() {
        //
        enthalpyLabel.setText("");
        entropyLabel.setText("");
        freeEnergyLabel.setText("");
        equilConstantLabel.setText("");
        percentCO2Label.setText("");
        percentH2Label.setText("");
        percentCOLabel.setText("");
        percentH2OLabel.setText("");
    }

    /** This method is called from within the constructor to
     * initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is
     * always regenerated by the Form Editor.
     */
    @SuppressWarnings("unchecked")
  // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
  private void initComponents() {

    mainPanel = new javax.swing.JPanel();
    jPanel1 = new javax.swing.JPanel();
    jLabel11 = new javax.swing.JLabel();
    jLabel12 = new javax.swing.JLabel();
    jLabel13 = new javax.swing.JLabel();
    feedRatioTextField = new javax.swing.JFormattedTextField();
    pressureTextField = new javax.swing.JFormattedTextField();
    temperatureTextField = new javax.swing.JFormattedTextField();
    equilibrateButton = new javax.swing.JButton();
    jLabel14 = new javax.swing.JLabel();
    jPanel2 = new javax.swing.JPanel();
    enthalpyLabel = new javax.swing.JLabel();
    entropyLabel = new javax.swing.JLabel();
    freeEnergyLabel = new javax.swing.JLabel();
    equilConstantLabel = new javax.swing.JLabel();
    jPanel3 = new javax.swing.JPanel();
    jLabel15 = new javax.swing.JLabel();
    jLabel16 = new javax.swing.JLabel();
    jLabel17 = new javax.swing.JLabel();
    jLabel18 = new javax.swing.JLabel();
    percentCO2Label = new javax.swing.JLabel();
    percentH2Label = new javax.swing.JLabel();
    percentCOLabel = new javax.swing.JLabel();
    percentH2OLabel = new javax.swing.JLabel();
    conversionLabel = new javax.swing.JLabel();
    menuBar = new javax.swing.JMenuBar();
    javax.swing.JMenu fileMenu = new javax.swing.JMenu();
    javax.swing.JMenuItem exitMenuItem = new javax.swing.JMenuItem();
    javax.swing.JMenu helpMenu = new javax.swing.JMenu();
    javax.swing.JMenuItem aboutMenuItem = new javax.swing.JMenuItem();
    jMenuItem5 = new javax.swing.JMenuItem();
    statusPanel = new javax.swing.JPanel();
    javax.swing.JSeparator statusPanelSeparator = new javax.swing.JSeparator();
    statusMessageLabel = new javax.swing.JLabel();
    statusAnimationLabel = new javax.swing.JLabel();
    progressBar = new javax.swing.JProgressBar();

    mainPanel.setName("mainPanel"); // NOI18N

    org.jdesktop.application.ResourceMap resourceMap = org.jdesktop.application.Application.getInstance(rwgsequilibrator.RWGSEquilibratorApp.class).getContext().getResourceMap(RWGSEquilibratorView.class);
    jPanel1.setBorder(javax.swing.BorderFactory.createTitledBorder(null, resourceMap.getString("jPanel1.border.title"), javax.swing.border.TitledBorder.DEFAULT_JUSTIFICATION, javax.swing.border.TitledBorder.DEFAULT_POSITION, resourceMap.getFont("jPanel1.border.titleFont"))); // NOI18N
    jPanel1.setForeground(resourceMap.getColor("jPanel1.foreground")); // NOI18N
    jPanel1.setName("jPanel1"); // NOI18N

    jLabel11.setHorizontalAlignment(javax.swing.SwingConstants.LEFT);
    jLabel11.setText(resourceMap.getString("jLabel11.text")); // NOI18N
    jLabel11.setName("jLabel11"); // NOI18N

    jLabel12.setHorizontalAlignment(javax.swing.SwingConstants.LEFT);
    jLabel12.setText(resourceMap.getString("jLabel12.text")); // NOI18N
    jLabel12.setName("jLabel12"); // NOI18N

    jLabel13.setHorizontalAlignment(javax.swing.SwingConstants.LEFT);
    jLabel13.setText(resourceMap.getString("jLabel13.text")); // NOI18N
    jLabel13.setName("jLabel13"); // NOI18N

    feedRatioTextField.setFormatterFactory(new javax.swing.text.DefaultFormatterFactory(new javax.swing.text.NumberFormatter()));
    feedRatioTextField.setText(resourceMap.getString("feedRatioTextField.text")); // NOI18N
    feedRatioTextField.setToolTipText(resourceMap.getString("feedRatioTextField.toolTipText")); // NOI18N
    feedRatioTextField.setName("feedRatioTextField"); // NOI18N
    feedRatioTextField.addPropertyChangeListener(new java.beans.PropertyChangeListener() {
      public void propertyChange(java.beans.PropertyChangeEvent evt) {
        newInitialFeedRatio(evt);
      }
    });

    pressureTextField.setFormatterFactory(new javax.swing.text.DefaultFormatterFactory(new javax.swing.text.NumberFormatter(new java.text.DecimalFormat("#0"))));
    pressureTextField.setText(resourceMap.getString("pressureTextField.text")); // NOI18N
    pressureTextField.setToolTipText(resourceMap.getString("pressureTextField.toolTipText")); // NOI18N
    pressureTextField.setName("pressureTextField"); // NOI18N
    pressureTextField.addPropertyChangeListener(new java.beans.PropertyChangeListener() {
      public void propertyChange(java.beans.PropertyChangeEvent evt) {
        newInitialPressure(evt);
      }
    });

    temperatureTextField.setFormatterFactory(new javax.swing.text.DefaultFormatterFactory(new javax.swing.text.NumberFormatter(new java.text.DecimalFormat("#0"))));
    temperatureTextField.setText(resourceMap.getString("temperatureTextField.text")); // NOI18N
    temperatureTextField.setToolTipText(resourceMap.getString("temperatureTextField.toolTipText")); // NOI18N
    temperatureTextField.setName("temperatureTextField"); // NOI18N
    temperatureTextField.setFocusLostBehavior(temperatureTextField.COMMIT_OR_REVERT);
    temperatureTextField.addPropertyChangeListener(new java.beans.PropertyChangeListener() {
      public void propertyChange(java.beans.PropertyChangeEvent evt) {
        newInitialTemperature(evt);
      }
    });

    org.jdesktop.layout.GroupLayout jPanel1Layout = new org.jdesktop.layout.GroupLayout(jPanel1);
    jPanel1.setLayout(jPanel1Layout);
    jPanel1Layout.setHorizontalGroup(
      jPanel1Layout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING)
      .add(jPanel1Layout.createSequentialGroup()
        .add(jPanel1Layout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING)
          .add(jPanel1Layout.createSequentialGroup()
            .add(jLabel12)
            .addPreferredGap(org.jdesktop.layout.LayoutStyle.RELATED)
            .add(pressureTextField, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, 81, Short.MAX_VALUE))
          .add(jPanel1Layout.createSequentialGroup()
            .add(jLabel13)
            .addPreferredGap(org.jdesktop.layout.LayoutStyle.RELATED)
            .add(feedRatioTextField, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, 81, Short.MAX_VALUE))
          .add(jPanel1Layout.createSequentialGroup()
            .add(jLabel11)
            .addPreferredGap(org.jdesktop.layout.LayoutStyle.RELATED, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
            .add(temperatureTextField, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE, 71, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE)))
        .addContainerGap())
    );

    jPanel1Layout.linkSize(new java.awt.Component[] {jLabel12, jLabel13}, org.jdesktop.layout.GroupLayout.HORIZONTAL);

    jPanel1Layout.setVerticalGroup(
      jPanel1Layout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING)
      .add(jPanel1Layout.createSequentialGroup()
        .add(jPanel1Layout.createParallelGroup(org.jdesktop.layout.GroupLayout.BASELINE)
          .add(jLabel11)
          .add(temperatureTextField, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE))
        .add(18, 18, 18)
        .add(jPanel1Layout.createParallelGroup(org.jdesktop.layout.GroupLayout.BASELINE)
          .add(jLabel12)
          .add(pressureTextField, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE))
        .add(18, 18, 18)
        .add(jPanel1Layout.createParallelGroup(org.jdesktop.layout.GroupLayout.BASELINE)
          .add(jLabel13)
          .add(feedRatioTextField, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE))
        .addContainerGap(org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE))
    );

    javax.swing.ActionMap actionMap = org.jdesktop.application.Application.getInstance(rwgsequilibrator.RWGSEquilibratorApp.class).getContext().getActionMap(RWGSEquilibratorView.class, this);
    equilibrateButton.setAction(actionMap.get("equilibrate")); // NOI18N
    equilibrateButton.setText(resourceMap.getString("equilibrateButton.text")); // NOI18N
    equilibrateButton.setName("equilibrateButton"); // NOI18N

    jLabel14.setHorizontalAlignment(javax.swing.SwingConstants.CENTER);
    jLabel14.setText(resourceMap.getString("jLabel14.text")); // NOI18N
    jLabel14.setName("jLabel14"); // NOI18N

    jPanel2.setBorder(javax.swing.BorderFactory.createTitledBorder(null, resourceMap.getString("jPanel2.border.title"), javax.swing.border.TitledBorder.DEFAULT_JUSTIFICATION, javax.swing.border.TitledBorder.DEFAULT_POSITION, resourceMap.getFont("jPanel2.border.titleFont"))); // NOI18N
    jPanel2.setName("jPanel2"); // NOI18N

    enthalpyLabel.setHorizontalAlignment(javax.swing.SwingConstants.CENTER);
    enthalpyLabel.setText(resourceMap.getString("enthalpyLabel.text")); // NOI18N
    enthalpyLabel.setName("enthalpyLabel"); // NOI18N

    entropyLabel.setHorizontalAlignment(javax.swing.SwingConstants.CENTER);
    entropyLabel.setText(resourceMap.getString("entropyLabel.text")); // NOI18N
    entropyLabel.setName("entropyLabel"); // NOI18N

    freeEnergyLabel.setHorizontalAlignment(javax.swing.SwingConstants.CENTER);
    freeEnergyLabel.setText(resourceMap.getString("freeEnergyLabel.text")); // NOI18N
    freeEnergyLabel.setName("freeEnergyLabel"); // NOI18N

    equilConstantLabel.setHorizontalAlignment(javax.swing.SwingConstants.CENTER);
    equilConstantLabel.setText(resourceMap.getString("equilConstantLabel.text")); // NOI18N
    equilConstantLabel.setName("equilConstantLabel"); // NOI18N

    org.jdesktop.layout.GroupLayout jPanel2Layout = new org.jdesktop.layout.GroupLayout(jPanel2);
    jPanel2.setLayout(jPanel2Layout);
    jPanel2Layout.setHorizontalGroup(
      jPanel2Layout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING)
      .add(enthalpyLabel, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, 212, Short.MAX_VALUE)
      .add(org.jdesktop.layout.GroupLayout.TRAILING, entropyLabel, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, 212, Short.MAX_VALUE)
      .add(org.jdesktop.layout.GroupLayout.TRAILING, freeEnergyLabel, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, 212, Short.MAX_VALUE)
      .add(equilConstantLabel, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, 212, Short.MAX_VALUE)
    );
    jPanel2Layout.setVerticalGroup(
      jPanel2Layout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING)
      .add(jPanel2Layout.createSequentialGroup()
        .add(enthalpyLabel)
        .add(18, 18, 18)
        .add(entropyLabel)
        .add(18, 18, 18)
        .add(freeEnergyLabel)
        .addPreferredGap(org.jdesktop.layout.LayoutStyle.RELATED, 20, Short.MAX_VALUE)
        .add(equilConstantLabel)
        .addContainerGap())
    );

    jPanel3.setBorder(javax.swing.BorderFactory.createTitledBorder(null, resourceMap.getString("jPanel3.border.title"), javax.swing.border.TitledBorder.DEFAULT_JUSTIFICATION, javax.swing.border.TitledBorder.DEFAULT_POSITION, resourceMap.getFont("jPanel3.border.titleFont"))); // NOI18N
    jPanel3.setName("jPanel3"); // NOI18N

    jLabel15.setText(resourceMap.getString("jLabel15.text")); // NOI18N
    jLabel15.setName("jLabel15"); // NOI18N

    jLabel16.setText(resourceMap.getString("jLabel16.text")); // NOI18N
    jLabel16.setName("jLabel16"); // NOI18N

    jLabel17.setText(resourceMap.getString("jLabel17.text")); // NOI18N
    jLabel17.setName("jLabel17"); // NOI18N

    jLabel18.setText(resourceMap.getString("jLabel18.text")); // NOI18N
    jLabel18.setName("jLabel18"); // NOI18N

    percentCO2Label.setHorizontalAlignment(javax.swing.SwingConstants.LEFT);
    percentCO2Label.setText(resourceMap.getString("percentCO2Label.text")); // NOI18N
    percentCO2Label.setHorizontalTextPosition(javax.swing.SwingConstants.LEFT);
    percentCO2Label.setName("percentCO2Label"); // NOI18N

    percentH2Label.setHorizontalAlignment(javax.swing.SwingConstants.LEFT);
    percentH2Label.setText(resourceMap.getString("percentH2Label.text")); // NOI18N
    percentH2Label.setHorizontalTextPosition(javax.swing.SwingConstants.LEFT);
    percentH2Label.setName("percentH2Label"); // NOI18N

    percentCOLabel.setHorizontalAlignment(javax.swing.SwingConstants.LEFT);
    percentCOLabel.setText(resourceMap.getString("percentCOLabel.text")); // NOI18N
    percentCOLabel.setHorizontalTextPosition(javax.swing.SwingConstants.LEFT);
    percentCOLabel.setName("percentCOLabel"); // NOI18N

    percentH2OLabel.setHorizontalAlignment(javax.swing.SwingConstants.LEFT);
    percentH2OLabel.setText(resourceMap.getString("percentH2OLabel.text")); // NOI18N
    percentH2OLabel.setHorizontalTextPosition(javax.swing.SwingConstants.LEFT);
    percentH2OLabel.setName("percentH2OLabel"); // NOI18N

    org.jdesktop.layout.GroupLayout jPanel3Layout = new org.jdesktop.layout.GroupLayout(jPanel3);
    jPanel3.setLayout(jPanel3Layout);
    jPanel3Layout.setHorizontalGroup(
      jPanel3Layout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING)
      .add(jPanel3Layout.createSequentialGroup()
        .addContainerGap()
        .add(jPanel3Layout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING)
          .add(jPanel3Layout.createSequentialGroup()
            .add(jLabel15)
            .addPreferredGap(org.jdesktop.layout.LayoutStyle.RELATED)
            .add(percentCO2Label, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, 134, Short.MAX_VALUE))
          .add(jPanel3Layout.createSequentialGroup()
            .add(jLabel16)
            .addPreferredGap(org.jdesktop.layout.LayoutStyle.RELATED)
            .add(percentH2Label, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, 143, Short.MAX_VALUE))
          .add(jPanel3Layout.createSequentialGroup()
            .add(jLabel17)
            .addPreferredGap(org.jdesktop.layout.LayoutStyle.RELATED)
            .add(percentCOLabel, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, 141, Short.MAX_VALUE))
          .add(jPanel3Layout.createSequentialGroup()
            .add(jLabel18)
            .addPreferredGap(org.jdesktop.layout.LayoutStyle.RELATED)
            .add(percentH2OLabel, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, 133, Short.MAX_VALUE)))
        .addContainerGap())
    );
    jPanel3Layout.setVerticalGroup(
      jPanel3Layout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING)
      .add(jPanel3Layout.createSequentialGroup()
        .add(jPanel3Layout.createParallelGroup(org.jdesktop.layout.GroupLayout.BASELINE)
          .add(jLabel15)
          .add(percentCO2Label))
        .addPreferredGap(org.jdesktop.layout.LayoutStyle.RELATED)
        .add(jPanel3Layout.createParallelGroup(org.jdesktop.layout.GroupLayout.BASELINE)
          .add(jLabel16)
          .add(percentH2Label))
        .addPreferredGap(org.jdesktop.layout.LayoutStyle.RELATED)
        .add(jPanel3Layout.createParallelGroup(org.jdesktop.layout.GroupLayout.BASELINE)
          .add(jLabel17)
          .add(percentCOLabel))
        .addPreferredGap(org.jdesktop.layout.LayoutStyle.RELATED)
        .add(jPanel3Layout.createParallelGroup(org.jdesktop.layout.GroupLayout.BASELINE)
          .add(jLabel18)
          .add(percentH2OLabel)))
    );

    conversionLabel.setText(resourceMap.getString("conversionLabel.text")); // NOI18N
    conversionLabel.setName("conversionLabel"); // NOI18N

    org.jdesktop.layout.GroupLayout mainPanelLayout = new org.jdesktop.layout.GroupLayout(mainPanel);
    mainPanel.setLayout(mainPanelLayout);
    mainPanelLayout.setHorizontalGroup(
      mainPanelLayout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING)
      .add(mainPanelLayout.createSequentialGroup()
        .addContainerGap()
        .add(mainPanelLayout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING)
          .add(conversionLabel, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, 458, Short.MAX_VALUE)
          .add(mainPanelLayout.createSequentialGroup()
            .add(mainPanelLayout.createParallelGroup(org.jdesktop.layout.GroupLayout.TRAILING, false)
              .add(org.jdesktop.layout.GroupLayout.LEADING, jLabel14, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
              .add(org.jdesktop.layout.GroupLayout.LEADING, equilibrateButton, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
              .add(org.jdesktop.layout.GroupLayout.LEADING, jPanel1, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE))
            .add(18, 18, 18)
            .add(mainPanelLayout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING)
              .add(jPanel2, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
              .add(jPanel3, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE))))
        .addContainerGap())
    );
    mainPanelLayout.setVerticalGroup(
      mainPanelLayout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING)
      .add(mainPanelLayout.createSequentialGroup()
        .addContainerGap()
        .add(mainPanelLayout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING, false)
          .add(jPanel2, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
          .add(jPanel1, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE))
        .addPreferredGap(org.jdesktop.layout.LayoutStyle.RELATED, 7, Short.MAX_VALUE)
        .add(mainPanelLayout.createParallelGroup(org.jdesktop.layout.GroupLayout.CENTER)
          .add(mainPanelLayout.createSequentialGroup()
            .add(equilibrateButton)
            .addPreferredGap(org.jdesktop.layout.LayoutStyle.RELATED)
            .add(jLabel14))
          .add(jPanel3, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE))
        .addPreferredGap(org.jdesktop.layout.LayoutStyle.RELATED)
        .add(conversionLabel)
        .addContainerGap())
    );

    menuBar.setName("menuBar"); // NOI18N

    fileMenu.setText(resourceMap.getString("fileMenu.text")); // NOI18N
    fileMenu.setName("fileMenu"); // NOI18N

    exitMenuItem.setAction(actionMap.get("quit")); // NOI18N
    exitMenuItem.setName("exitMenuItem"); // NOI18N
    fileMenu.add(exitMenuItem);

    menuBar.add(fileMenu);

    helpMenu.setText(resourceMap.getString("helpMenu.text")); // NOI18N
    helpMenu.setName("helpMenu"); // NOI18N

    aboutMenuItem.setAction(actionMap.get("showAboutBox")); // NOI18N
    aboutMenuItem.setToolTipText(resourceMap.getString("aboutMenuItem.toolTipText")); // NOI18N
    aboutMenuItem.setName("aboutMenuItem"); // NOI18N
    helpMenu.add(aboutMenuItem);

    jMenuItem5.setAction(actionMap.get("showUserGuide")); // NOI18N
    jMenuItem5.setText(resourceMap.getString("jMenuItem5.text")); // NOI18N
    jMenuItem5.setToolTipText(resourceMap.getString("jMenuItem5.toolTipText")); // NOI18N
    jMenuItem5.setName("jMenuItem5"); // NOI18N
    helpMenu.add(jMenuItem5);

    menuBar.add(helpMenu);

    statusPanel.setName("statusPanel"); // NOI18N

    statusPanelSeparator.setName("statusPanelSeparator"); // NOI18N

    statusMessageLabel.setName("statusMessageLabel"); // NOI18N

    statusAnimationLabel.setHorizontalAlignment(javax.swing.SwingConstants.LEFT);
    statusAnimationLabel.setName("statusAnimationLabel"); // NOI18N

    progressBar.setName("progressBar"); // NOI18N

    org.jdesktop.layout.GroupLayout statusPanelLayout = new org.jdesktop.layout.GroupLayout(statusPanel);
    statusPanel.setLayout(statusPanelLayout);
    statusPanelLayout.setHorizontalGroup(
      statusPanelLayout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING)
      .add(statusPanelSeparator, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, 498, Short.MAX_VALUE)
      .add(org.jdesktop.layout.GroupLayout.TRAILING, statusPanelLayout.createSequentialGroup()
        .addContainerGap()
        .add(statusMessageLabel)
        .addPreferredGap(org.jdesktop.layout.LayoutStyle.RELATED, 304, Short.MAX_VALUE)
        .add(statusAnimationLabel)
        .addPreferredGap(org.jdesktop.layout.LayoutStyle.RELATED)
        .add(progressBar, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE)
        .addContainerGap())
    );
    statusPanelLayout.setVerticalGroup(
      statusPanelLayout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING)
      .add(statusPanelLayout.createSequentialGroup()
        .add(statusPanelSeparator, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE, 2, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE)
        .addPreferredGap(org.jdesktop.layout.LayoutStyle.RELATED, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
        .add(statusPanelLayout.createParallelGroup(org.jdesktop.layout.GroupLayout.TRAILING)
          .add(statusPanelLayout.createSequentialGroup()
            .add(statusPanelLayout.createParallelGroup(org.jdesktop.layout.GroupLayout.BASELINE)
              .add(statusMessageLabel)
              .add(statusAnimationLabel))
            .add(32, 32, 32))
          .add(statusPanelLayout.createSequentialGroup()
            .add(progressBar, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE)
            .addContainerGap())))
    );

    setComponent(mainPanel);
    setMenuBar(menuBar);
    setStatusBar(statusPanel);
  }// </editor-fold>//GEN-END:initComponents

    private void newInitialTemperature(java.beans.PropertyChangeEvent evt) {//GEN-FIRST:event_newInitialTemperature
        // see if the value has changed
        double newTemperature = ((Number)temperatureTextField.getValue()).doubleValue();
        if (newTemperature != T) {
            clearResults();
            if (newTemperature < 298.) {
                T = 298.;
            } else if (newTemperature > 1200.) {
                T = 1200.;
            } else {
                T = newTemperature;
            }
            temperatureTextField.setValue(new Double(T));
        }
    }//GEN-LAST:event_newInitialTemperature

    private void newInitialPressure(java.beans.PropertyChangeEvent evt) {//GEN-FIRST:event_newInitialPressure
        // see if the value has changed
        double newPressure = ((Number)pressureTextField.getValue()).doubleValue();
        if (newPressure != P) {
            clearResults();
            if (newPressure < 1.) {
                P = 1.;
            } else if (newPressure > 100.) {
                P = 100.;
            } else {
                P = newPressure;
            }
            pressureTextField.setValue(new Double(P));
        }
    }//GEN-LAST:event_newInitialPressure

    private void newInitialFeedRatio(java.beans.PropertyChangeEvent evt) {//GEN-FIRST:event_newInitialFeedRatio
        // see if the value has changed
        double newFeedRatio = ((Number)feedRatioTextField.getValue()).doubleValue();
        if (newFeedRatio != ratio) {
            clearResults();
            if (newFeedRatio < 0.01) {
                ratio = 0.01;
            } else if (newFeedRatio > 100.0) {
                ratio = 100.;
            } else {
                ratio = newFeedRatio;
            }
            feedRatioTextField.setValue(new Double(ratio));
        }
    }//GEN-LAST:event_newInitialFeedRatio


  // Variables declaration - do not modify//GEN-BEGIN:variables
  private javax.swing.JLabel conversionLabel;
  private javax.swing.JLabel enthalpyLabel;
  private javax.swing.JLabel entropyLabel;
  private javax.swing.JLabel equilConstantLabel;
  private javax.swing.JButton equilibrateButton;
  private javax.swing.JFormattedTextField feedRatioTextField;
  private javax.swing.JLabel freeEnergyLabel;
  private javax.swing.JLabel jLabel11;
  private javax.swing.JLabel jLabel12;
  private javax.swing.JLabel jLabel13;
  private javax.swing.JLabel jLabel14;
  private javax.swing.JLabel jLabel15;
  private javax.swing.JLabel jLabel16;
  private javax.swing.JLabel jLabel17;
  private javax.swing.JLabel jLabel18;
  private javax.swing.JMenuItem jMenuItem5;
  private javax.swing.JPanel jPanel1;
  private javax.swing.JPanel jPanel2;
  private javax.swing.JPanel jPanel3;
  private javax.swing.JPanel mainPanel;
  private javax.swing.JMenuBar menuBar;
  private javax.swing.JLabel percentCO2Label;
  private javax.swing.JLabel percentCOLabel;
  private javax.swing.JLabel percentH2Label;
  private javax.swing.JLabel percentH2OLabel;
  private javax.swing.JFormattedTextField pressureTextField;
  private javax.swing.JProgressBar progressBar;
  private javax.swing.JLabel statusAnimationLabel;
  private javax.swing.JLabel statusMessageLabel;
  private javax.swing.JPanel statusPanel;
  private javax.swing.JFormattedTextField temperatureTextField;
  // End of variables declaration//GEN-END:variables

    private final Timer messageTimer;
    private final Timer busyIconTimer;
    private final Icon idleIcon;
    private final Icon[] busyIcons = new Icon[15];
    private int busyIconIndex = 0;

    private JDialog aboutBox;
    private JDialog userGuide;
    private double T, P, ratio, n0CO2, n0H2;
    private Shomate CO2, H2, CO, H2O;
    private double percentCO2, percentH2, percentCO, percentH2O;
}
