/*
 * MeOHEquilibratorView.java
 */

package meohequilibrator;

import chem.ChemConstants;
import chem.Shomate;
import numeth.NMResult;
import org.jdesktop.application.Action;
import org.jdesktop.application.ResourceMap;
import org.jdesktop.application.SingleFrameApplication;
import org.jdesktop.application.FrameView;
import org.jdesktop.application.Task;
import java.text.DecimalFormat;
import javax.swing.JDialog;
import javax.swing.JFrame;
import numeth.NMJacEstEqns;
import numeth.NMJacEstimator;
import numeth.NMNewtRaphEqns;
import numeth.NMNewtRaphSolver;

/**
 * The application's main frame.
 */
public class MeOHEquilibratorView extends FrameView {

    public MeOHEquilibratorView(SingleFrameApplication app) {
        super(app);

        initComponents();

        // status bar initialization - message timeout, idle icon and busy animation, etc
        ResourceMap resourceMap = getResourceMap();

        // set default values for the initial conditions
        T = 400.;
        P = 1.0;
        ratio = 1.0;
        temperatureTextField.setValue(T);
        pressureTextField.setValue(P);
        feedRatioTextField.setValue(ratio);
        // instantiate a Shomate object for each species
        String[] H2Params = {
            "298	1500	33.107800	-11.508000	11.609300	-2.844400	-0.159665	-9.991971	172.788000	0.000000",
            "1500	6000	34.143400	0.503927	0.372036	-0.038599	-8.074761	-21.218800	162.093000	0.000000"
        };
        H2 = new Shomate(H2Params);
        String[] COParams = {
            "298	1300	25.567590	6.096130	4.054656	-2.671301	0.131021	-118.008900	227.366500	-110.527100",
            "1300	6000	35.150700	1.300095	-0.205921	0.013550	-3.282780	-127.837500	231.712000	-110.527100"
        };
        CO = new Shomate(COParams);
        String[] MeOHParams = {
            "298	6000	21.138073	70.878654	25.853554	-28.497905	0.000000	-210.789702	243.296143	-201.171528"
        };
        MeOH = new Shomate(MeOHParams);
    }

    // Action for the Help menu About item
    @Action
    public void showAboutBox() {
        if (aboutBox == null) {
            JFrame mainFrame = MeOHEquilibratorApp.getApplication().getMainFrame();
            aboutBox = new MeOHEquilibratorAboutBox(mainFrame);
            aboutBox.setLocationRelativeTo(mainFrame);
        }
        MeOHEquilibratorApp.getApplication().show(aboutBox);
    }

    // Action for the Help menu User Guide item
    @Action
    public void showUserGuide() {

        if (userGuide == null) {
            JFrame mainFrame = MeOHEquilibratorApp.getApplication().getMainFrame();
            userGuide = new MeOHEquilibratorUserGuide(mainFrame);
            userGuide.setLocationRelativeTo(mainFrame);
        }
        MeOHEquilibratorApp.getApplication().show(userGuide);
    }

    // Action for the Equilibrate button
    @Action
    public Task equilibrate() {
        return new EquilibrateTask(getApplication());
    }

    private class EquilibrateTask extends org.jdesktop.application.Task<Object, Void>
        implements NMNewtRaphEqns, NMJacEstEqns {
        double temperature, pressure, feedRatio, conversion;
        double sRxn, hRxn, gRxn, K;
        double[] x, y;
        EquilibrateTask(org.jdesktop.application.Application app) {
            // Runs on the EDT.  Copy GUI state that
            // doInBackground() depends on from parameters
            // to EquilibrateTask fields, here.
            super(app);

            // If this is the first click
                // see if they want to use a log file and if so, open it.

            // Get the initial conditions from the input textfields
            temperature = ((Number)temperatureTextField.getValue()).doubleValue();
            pressure = ((Number)pressureTextField.getValue()).doubleValue();
            feedRatio = ((Number)feedRatioTextField.getValue()).doubleValue();

            equilibrateButton.setEnabled(false);
        }
        @Override protected Object doInBackground() {
            // Your Task's code here.  This method runs
            // on a background thread, so don't reference
            // the Swing GUI from here.
            
            // Calculate initial moles and volume
            n0H2 = 1.0;
            n0CO = feedRatio;

            // Calculate the thermo properties of each species
            double sH2 = H2.sStd(temperature).getDoubleValue();
            double sCO = CO.sStd(temperature).getDoubleValue();
            double sMeOH = MeOH.sStd(temperature).getDoubleValue();
            double hH2 = H2.hForm(temperature).getDoubleValue();
            double hCO = CO.hForm(temperature).getDoubleValue();
            double hMeOH = MeOH.hForm(temperature).getDoubleValue();
            double gH2 = H2.gForm(temperature).getDoubleValue();
            double gCO = CO.gForm(temperature).getDoubleValue();
            double gMeOH = MeOH.gForm(temperature).getDoubleValue();

            // Calculate the thermo properties for the reaction
            sRxn = sMeOH - sCO - 2 * sH2;
            hRxn = hMeOH - hCO - 2 * hH2;
            gRxn = gMeOH - gCO - 2 * gH2;
            K = Math.exp(-gRxn*1000.0/ChemConstants.RgasE/temperature);

            // Calculate the equilibrium composition
            x = new double[3];
            y = new double[3];
            x[0] = n0CO;
            x[1] = n0H2;
            x[2] = 0.0;
            NMNewtRaphSolver theSolver = new NMNewtRaphSolver(this);
            theSolver.setTol(1.0e-4);
            theSolver.setTreatAsNonNegative(true);
            NMResult theResult = theSolver.solve(x, y);
            percentCO = 100*x[0]/(x[0] + x[1] + x[2]);
            percentH2 = 100*x[1]/(x[0] + x[1] + x[2]);
            percentMeOH = 100*x[2]/(x[0] + x[1] + x[2]);

            // Calculate the conversion
            if (feedRatio <= 0.5) {
              // CO is the limiting reagent
              conversion = 100*(n0CO - x[0])/n0CO;
            } else {
              // H2 is the limiting reagent
              conversion = 100*(n0H2 -x[1])/n0H2;
            }
            return null;  // return your result
        }
        @Override protected void succeeded(Object result) {
            // Runs on the EDT.  Update the GUI based on
            // the result computed by doInBackground().

            // Write the results to the screen
            DecimalFormat noDecimal = new DecimalFormat("####");
            DecimalFormat threeSigFigs = new DecimalFormat("0.00E00");
            DecimalFormat notScientific = new DecimalFormat("##0.00");
            String txt = "<html>&#916;S(" + noDecimal.format(temperature);
            txt += ") = ";
            if (Math.abs(sRxn) < 0.1) {
                txt += threeSigFigs.format(sRxn) + " J/(mol K)";
            } else {
                txt += notScientific.format(sRxn) + " J/(mol K)";
            }
            entropyLabel.setText(txt);
            txt = "<html>&#916;H(" + noDecimal.format(temperature);
            txt += ") = ";
            if (Math.abs(hRxn) < 0.1) {
                txt += threeSigFigs.format(hRxn) + " kJ/mol";
            } else {
                txt += notScientific.format(hRxn) + " kJ/mol";
            }
            enthalpyLabel.setText(txt);
            txt = "<html>&#916;G(" + noDecimal.format(temperature);
            txt += ") = ";
            if (Math.abs(gRxn) < 0.1) {
                txt += threeSigFigs.format(gRxn) + " kJ/mol";
            } else {
                txt += notScientific.format(gRxn) + " kJ/mol";
            }
            freeEnergyLabel.setText(txt);
            txt = "<html>K(" + noDecimal.format(temperature);
            txt += ") = " + threeSigFigs.format(K);
            equilConstantLabel.setText(txt);
            if (percentCO <0.1) {
                txt = threeSigFigs.format(percentCO);
            } else {
                txt = notScientific.format(percentCO);
            }
            txt += " %";
            percentCOLabel.setText(txt);
            if (percentH2 <0.1) {
                txt = threeSigFigs.format(percentH2); 
            } else {
                txt = notScientific.format(percentH2);
            }
            txt += " %";
            percentH2Label.setText(txt);
            if (percentMeOH <0.1) {
                txt = threeSigFigs.format(percentMeOH);
            } else {
                txt = notScientific.format(percentMeOH);
            }
            txt += " %";
            percentMeOHLabel.setText(txt);

            equilibrateButton.setEnabled(true);

            // set the conversion label

            txt= "<html>Limiting Reagent ";
            if (feedRatio < 0.5) {
                txt += "(CO) ";
            }
            if (feedRatio > 0.5) {
                txt += "(H<sub>2</sub>) ";
            }
            txt += "Conversion is ";
            if (conversion <0.1) {
                txt += threeSigFigs.format(conversion);
            } else {
                txt += notScientific.format(conversion);
            }
            txt += " %";
            conversionLabel.setText(txt);

            equilibrateButton.setEnabled(true);
        }

        public NMResult evalNewtRaphEqns(double[] x, double[] y) {
            NMResult theResult = new NMResult();
            // 0 = CO, 1 = H2, 2 = MeOH
            y[0] = K*x[0]*x[1]*x[1]*pressure*pressure - x[2]*(x[0]+x[1]+x[2])*(x[0]+x[1]+x[2]);
            y[1] = n0CO - x[2]- x[0];
            y[2] = 2*n0H2 - 2*x[1] - 4*x[2];
            return theResult;
        }

        public NMResult evalNewtRaphJac(double[] x, double[][] jac) {
            NMJacEstimator jacEstimator = new NMJacEstimator(this);
            NMResult theResult = new NMResult();
            jacEstimator.estimate(3, x, jac, 1.0E-4);
            return theResult;
        }

        public NMResult evalJacEstEqns(double[] x, double[] y) {
            NMResult theResult = evalNewtRaphEqns(x,y);
            return theResult;
        }
    }

    public void clearResults() {
        //
        enthalpyLabel.setText("");
        entropyLabel.setText("");
        freeEnergyLabel.setText("");
        equilConstantLabel.setText("");
        percentCOLabel.setText("");
        percentH2Label.setText("");
        percentCOLabel.setText("");
        percentMeOHLabel.setText("");
    }

    /** This method is called from within the constructor to
     * initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is
     * always regenerated by the Form Editor.
     */
    @SuppressWarnings("unchecked")
  // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
  private void initComponents() {

    mainPanel = new javax.swing.JPanel();
    jPanel1 = new javax.swing.JPanel();
    jLabel11 = new javax.swing.JLabel();
    jLabel12 = new javax.swing.JLabel();
    jLabel13 = new javax.swing.JLabel();
    feedRatioTextField = new javax.swing.JFormattedTextField();
    pressureTextField = new javax.swing.JFormattedTextField();
    temperatureTextField = new javax.swing.JFormattedTextField();
    equilibrateButton = new javax.swing.JButton();
    jLabel14 = new javax.swing.JLabel();
    jPanel2 = new javax.swing.JPanel();
    enthalpyLabel = new javax.swing.JLabel();
    entropyLabel = new javax.swing.JLabel();
    freeEnergyLabel = new javax.swing.JLabel();
    equilConstantLabel = new javax.swing.JLabel();
    jPanel3 = new javax.swing.JPanel();
    jLabel15 = new javax.swing.JLabel();
    jLabel16 = new javax.swing.JLabel();
    jLabel18 = new javax.swing.JLabel();
    percentCOLabel = new javax.swing.JLabel();
    percentH2Label = new javax.swing.JLabel();
    percentMeOHLabel = new javax.swing.JLabel();
    conversionLabel = new javax.swing.JLabel();
    menuBar = new javax.swing.JMenuBar();
    javax.swing.JMenu fileMenu = new javax.swing.JMenu();
    javax.swing.JMenuItem exitMenuItem = new javax.swing.JMenuItem();
    javax.swing.JMenu helpMenu = new javax.swing.JMenu();
    javax.swing.JMenuItem aboutMenuItem = new javax.swing.JMenuItem();
    jMenuItem5 = new javax.swing.JMenuItem();

    mainPanel.setName("mainPanel"); // NOI18N

    jPanel1.setBorder(javax.swing.BorderFactory.createTitledBorder("Initial Conditions"));
    jPanel1.setName("jPanel1"); // NOI18N

    jLabel11.setHorizontalAlignment(javax.swing.SwingConstants.LEFT);
    org.jdesktop.application.ResourceMap resourceMap = org.jdesktop.application.Application.getInstance(meohequilibrator.MeOHEquilibratorApp.class).getContext().getResourceMap(MeOHEquilibratorView.class);
    jLabel11.setText(resourceMap.getString("jLabel11.text")); // NOI18N
    jLabel11.setName("jLabel11"); // NOI18N

    jLabel12.setHorizontalAlignment(javax.swing.SwingConstants.LEFT);
    jLabel12.setText(resourceMap.getString("jLabel12.text")); // NOI18N
    jLabel12.setName("jLabel12"); // NOI18N

    jLabel13.setHorizontalAlignment(javax.swing.SwingConstants.LEFT);
    jLabel13.setText(resourceMap.getString("jLabel13.text")); // NOI18N
    jLabel13.setName("jLabel13"); // NOI18N

    feedRatioTextField.setFormatterFactory(new javax.swing.text.DefaultFormatterFactory(new javax.swing.text.NumberFormatter()));
    feedRatioTextField.setName("feedRatioTextField"); // NOI18N
    feedRatioTextField.addPropertyChangeListener(new java.beans.PropertyChangeListener() {
      public void propertyChange(java.beans.PropertyChangeEvent evt) {
        newInitialFeedRatio(evt);
      }
    });

    pressureTextField.setFormatterFactory(new javax.swing.text.DefaultFormatterFactory(new javax.swing.text.NumberFormatter(new java.text.DecimalFormat("#0"))));
    pressureTextField.setName("pressureTextField"); // NOI18N
    pressureTextField.addPropertyChangeListener(new java.beans.PropertyChangeListener() {
      public void propertyChange(java.beans.PropertyChangeEvent evt) {
        newInitialPressure(evt);
      }
    });

    temperatureTextField.setFormatterFactory(new javax.swing.text.DefaultFormatterFactory(new javax.swing.text.NumberFormatter(new java.text.DecimalFormat("#0"))));
    temperatureTextField.setName("temperatureTextField"); // NOI18N
    temperatureTextField.setFocusLostBehavior(temperatureTextField.COMMIT_OR_REVERT);
    temperatureTextField.addPropertyChangeListener(new java.beans.PropertyChangeListener() {
      public void propertyChange(java.beans.PropertyChangeEvent evt) {
        newInitialTemperature(evt);
      }
    });

    org.jdesktop.layout.GroupLayout jPanel1Layout = new org.jdesktop.layout.GroupLayout(jPanel1);
    jPanel1.setLayout(jPanel1Layout);
    jPanel1Layout.setHorizontalGroup(
      jPanel1Layout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING)
      .add(jPanel1Layout.createSequentialGroup()
        .add(jPanel1Layout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING)
          .add(jPanel1Layout.createSequentialGroup()
            .add(jLabel12)
            .addPreferredGap(org.jdesktop.layout.LayoutStyle.RELATED)
            .add(pressureTextField, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, 98, Short.MAX_VALUE))
          .add(jPanel1Layout.createSequentialGroup()
            .add(jLabel13)
            .addPreferredGap(org.jdesktop.layout.LayoutStyle.RELATED)
            .add(feedRatioTextField, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, 98, Short.MAX_VALUE))
          .add(jPanel1Layout.createSequentialGroup()
            .add(jLabel11)
            .addPreferredGap(org.jdesktop.layout.LayoutStyle.RELATED, 27, Short.MAX_VALUE)
            .add(temperatureTextField, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE, 71, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE)))
        .addContainerGap())
    );

    jPanel1Layout.linkSize(new java.awt.Component[] {jLabel12, jLabel13}, org.jdesktop.layout.GroupLayout.HORIZONTAL);

    jPanel1Layout.setVerticalGroup(
      jPanel1Layout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING)
      .add(jPanel1Layout.createSequentialGroup()
        .add(jPanel1Layout.createParallelGroup(org.jdesktop.layout.GroupLayout.BASELINE)
          .add(jLabel11)
          .add(temperatureTextField, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE))
        .add(18, 18, 18)
        .add(jPanel1Layout.createParallelGroup(org.jdesktop.layout.GroupLayout.BASELINE)
          .add(jLabel12)
          .add(pressureTextField, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE))
        .add(18, 18, 18)
        .add(jPanel1Layout.createParallelGroup(org.jdesktop.layout.GroupLayout.BASELINE)
          .add(jLabel13)
          .add(feedRatioTextField, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE))
        .addContainerGap(org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE))
    );

    javax.swing.ActionMap actionMap = org.jdesktop.application.Application.getInstance(meohequilibrator.MeOHEquilibratorApp.class).getContext().getActionMap(MeOHEquilibratorView.class, this);
    equilibrateButton.setAction(actionMap.get("equilibrate")); // NOI18N
    equilibrateButton.setName("equilibrateButton"); // NOI18N

    jLabel14.setHorizontalAlignment(javax.swing.SwingConstants.CENTER);
    jLabel14.setText(resourceMap.getString("jLabel14.text")); // NOI18N
    jLabel14.setName("jLabel14"); // NOI18N

    jPanel2.setBorder(javax.swing.BorderFactory.createTitledBorder(resourceMap.getString("jPanel2.border.title"))); // NOI18N
    jPanel2.setName("jPanel2"); // NOI18N

    enthalpyLabel.setHorizontalAlignment(javax.swing.SwingConstants.CENTER);
    enthalpyLabel.setText(resourceMap.getString("enthalpyLabel.text")); // NOI18N
    enthalpyLabel.setName("enthalpyLabel"); // NOI18N

    entropyLabel.setHorizontalAlignment(javax.swing.SwingConstants.CENTER);
    entropyLabel.setText(resourceMap.getString("entropyLabel.text")); // NOI18N
    entropyLabel.setName("entropyLabel"); // NOI18N

    freeEnergyLabel.setHorizontalAlignment(javax.swing.SwingConstants.CENTER);
    freeEnergyLabel.setText(resourceMap.getString("freeEnergyLabel.text")); // NOI18N
    freeEnergyLabel.setName("freeEnergyLabel"); // NOI18N

    equilConstantLabel.setHorizontalAlignment(javax.swing.SwingConstants.CENTER);
    equilConstantLabel.setText(resourceMap.getString("equilConstantLabel.text")); // NOI18N
    equilConstantLabel.setName("equilConstantLabel"); // NOI18N

    org.jdesktop.layout.GroupLayout jPanel2Layout = new org.jdesktop.layout.GroupLayout(jPanel2);
    jPanel2.setLayout(jPanel2Layout);
    jPanel2Layout.setHorizontalGroup(
      jPanel2Layout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING)
      .add(enthalpyLabel, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, 205, Short.MAX_VALUE)
      .add(org.jdesktop.layout.GroupLayout.TRAILING, entropyLabel, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, 205, Short.MAX_VALUE)
      .add(org.jdesktop.layout.GroupLayout.TRAILING, freeEnergyLabel, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, 205, Short.MAX_VALUE)
      .add(equilConstantLabel, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, 205, Short.MAX_VALUE)
    );
    jPanel2Layout.setVerticalGroup(
      jPanel2Layout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING)
      .add(jPanel2Layout.createSequentialGroup()
        .add(enthalpyLabel)
        .add(18, 18, 18)
        .add(entropyLabel)
        .add(18, 18, 18)
        .add(freeEnergyLabel)
        .addPreferredGap(org.jdesktop.layout.LayoutStyle.RELATED, 20, Short.MAX_VALUE)
        .add(equilConstantLabel)
        .addContainerGap())
    );

    jPanel3.setBorder(javax.swing.BorderFactory.createTitledBorder("Equilibrium Mole Percent"));
    jPanel3.setName("jPanel3"); // NOI18N

    jLabel15.setText(resourceMap.getString("jLabel15.text")); // NOI18N
    jLabel15.setName("jLabel15"); // NOI18N

    jLabel16.setText(resourceMap.getString("jLabel16.text")); // NOI18N
    jLabel16.setName("jLabel16"); // NOI18N

    jLabel18.setText(resourceMap.getString("jLabel18.text")); // NOI18N
    jLabel18.setName("jLabel18"); // NOI18N

    percentCOLabel.setHorizontalAlignment(javax.swing.SwingConstants.LEFT);
    percentCOLabel.setText(resourceMap.getString("percentCOLabel.text")); // NOI18N
    percentCOLabel.setHorizontalTextPosition(javax.swing.SwingConstants.LEFT);
    percentCOLabel.setName("percentCOLabel"); // NOI18N

    percentH2Label.setHorizontalAlignment(javax.swing.SwingConstants.LEFT);
    percentH2Label.setText(resourceMap.getString("percentH2Label.text")); // NOI18N
    percentH2Label.setHorizontalTextPosition(javax.swing.SwingConstants.LEFT);
    percentH2Label.setName("percentH2Label"); // NOI18N

    percentMeOHLabel.setHorizontalAlignment(javax.swing.SwingConstants.LEFT);
    percentMeOHLabel.setText(resourceMap.getString("percentMeOHLabel.text")); // NOI18N
    percentMeOHLabel.setHorizontalTextPosition(javax.swing.SwingConstants.LEFT);
    percentMeOHLabel.setName("percentMeOHLabel"); // NOI18N

    org.jdesktop.layout.GroupLayout jPanel3Layout = new org.jdesktop.layout.GroupLayout(jPanel3);
    jPanel3.setLayout(jPanel3Layout);
    jPanel3Layout.setHorizontalGroup(
      jPanel3Layout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING)
      .add(jPanel3Layout.createSequentialGroup()
        .addContainerGap()
        .add(jPanel3Layout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING)
          .add(jPanel3Layout.createSequentialGroup()
            .add(jLabel15)
            .addPreferredGap(org.jdesktop.layout.LayoutStyle.RELATED)
            .add(percentCOLabel, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, 134, Short.MAX_VALUE))
          .add(jPanel3Layout.createSequentialGroup()
            .add(jLabel16)
            .addPreferredGap(org.jdesktop.layout.LayoutStyle.RELATED)
            .add(percentH2Label, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, 136, Short.MAX_VALUE))
          .add(jPanel3Layout.createSequentialGroup()
            .add(jLabel18)
            .add(18, 18, 18)
            .add(percentMeOHLabel, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, 97, Short.MAX_VALUE)))
        .addContainerGap())
    );
    jPanel3Layout.setVerticalGroup(
      jPanel3Layout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING)
      .add(jPanel3Layout.createSequentialGroup()
        .add(jPanel3Layout.createParallelGroup(org.jdesktop.layout.GroupLayout.BASELINE)
          .add(jLabel15)
          .add(percentCOLabel))
        .addPreferredGap(org.jdesktop.layout.LayoutStyle.RELATED)
        .add(jPanel3Layout.createParallelGroup(org.jdesktop.layout.GroupLayout.BASELINE)
          .add(jLabel16)
          .add(percentH2Label))
        .addPreferredGap(org.jdesktop.layout.LayoutStyle.RELATED)
        .add(jPanel3Layout.createParallelGroup(org.jdesktop.layout.GroupLayout.BASELINE)
          .add(jLabel18)
          .add(percentMeOHLabel))
        .add(20, 20, 20))
    );

    conversionLabel.setText(resourceMap.getString("conversionLabel.text")); // NOI18N
    conversionLabel.setName("conversionLabel"); // NOI18N

    org.jdesktop.layout.GroupLayout mainPanelLayout = new org.jdesktop.layout.GroupLayout(mainPanel);
    mainPanel.setLayout(mainPanelLayout);
    mainPanelLayout.setHorizontalGroup(
      mainPanelLayout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING)
      .add(mainPanelLayout.createSequentialGroup()
        .addContainerGap()
        .add(mainPanelLayout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING)
          .add(mainPanelLayout.createSequentialGroup()
            .add(mainPanelLayout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING)
              .add(jPanel1, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE)
              .add(equilibrateButton, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, 233, Short.MAX_VALUE)
              .add(jLabel14, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, 233, Short.MAX_VALUE))
            .addPreferredGap(org.jdesktop.layout.LayoutStyle.RELATED)
            .add(mainPanelLayout.createParallelGroup(org.jdesktop.layout.GroupLayout.TRAILING, false)
              .add(jPanel3, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
              .add(jPanel2, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)))
          .add(conversionLabel, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, 459, Short.MAX_VALUE))
        .addContainerGap())
    );
    mainPanelLayout.setVerticalGroup(
      mainPanelLayout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING)
      .add(mainPanelLayout.createSequentialGroup()
        .addContainerGap()
        .add(mainPanelLayout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING)
          .add(jPanel2, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
          .add(mainPanelLayout.createSequentialGroup()
            .add(jPanel1, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE)
            .addPreferredGap(org.jdesktop.layout.LayoutStyle.RELATED)))
        .add(mainPanelLayout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING)
          .add(mainPanelLayout.createSequentialGroup()
            .add(24, 24, 24)
            .add(equilibrateButton)
            .add(18, 18, 18)
            .add(jLabel14))
          .add(mainPanelLayout.createSequentialGroup()
            .addPreferredGap(org.jdesktop.layout.LayoutStyle.UNRELATED)
            .add(jPanel3, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE)))
        .addPreferredGap(org.jdesktop.layout.LayoutStyle.RELATED)
        .add(conversionLabel)
        .addContainerGap())
    );

    jPanel2.getAccessibleContext().setAccessibleName(resourceMap.getString("jPanel2.AccessibleContext.accessibleName")); // NOI18N

    menuBar.setName("menuBar"); // NOI18N

    fileMenu.setText(resourceMap.getString("fileMenu.text")); // NOI18N
    fileMenu.setName("fileMenu"); // NOI18N

    exitMenuItem.setAction(actionMap.get("quit")); // NOI18N
    exitMenuItem.setName("exitMenuItem"); // NOI18N
    fileMenu.add(exitMenuItem);

    menuBar.add(fileMenu);

    helpMenu.setText(resourceMap.getString("helpMenu.text")); // NOI18N
    helpMenu.setName("helpMenu"); // NOI18N

    aboutMenuItem.setAction(actionMap.get("showAboutBox")); // NOI18N
    aboutMenuItem.setName("aboutMenuItem"); // NOI18N
    helpMenu.add(aboutMenuItem);

    jMenuItem5.setAction(actionMap.get("showUserGuide")); // NOI18N
    jMenuItem5.setName("jMenuItem5"); // NOI18N
    helpMenu.add(jMenuItem5);

    menuBar.add(helpMenu);

    setComponent(mainPanel);
    setMenuBar(menuBar);
  }// </editor-fold>//GEN-END:initComponents

    private void newInitialTemperature(java.beans.PropertyChangeEvent evt) {//GEN-FIRST:event_newInitialTemperature
        // see if the value has changed
        double newTemperature = ((Number)temperatureTextField.getValue()).doubleValue();
        if (newTemperature != T) {
            clearResults();
            if (newTemperature < 298.) {
                T = 298.;
            } else if (newTemperature > 1200.) {
                T = 1200.;
            } else {
                T = newTemperature;
            }
            temperatureTextField.setValue(new Double(T));
        }
    }//GEN-LAST:event_newInitialTemperature

    private void newInitialPressure(java.beans.PropertyChangeEvent evt) {//GEN-FIRST:event_newInitialPressure
        // see if the value has changed
        double newPressure = ((Number)pressureTextField.getValue()).doubleValue();
        if (newPressure != P) {
            clearResults();
            if (newPressure < 1.) {
                P = 1.;
            } else if (newPressure > 100.) {
                P = 100.;
            } else {
                P = newPressure;
            }
            pressureTextField.setValue(new Double(P));
        }
    }//GEN-LAST:event_newInitialPressure

    private void newInitialFeedRatio(java.beans.PropertyChangeEvent evt) {//GEN-FIRST:event_newInitialFeedRatio
        // see if the value has changed
        double newFeedRatio = ((Number)feedRatioTextField.getValue()).doubleValue();
        if (newFeedRatio != ratio) {
            clearResults();
            if (newFeedRatio < 0.01) {
                ratio = 0.01;
            } else if (newFeedRatio > 100.0) {
                ratio = 100.;
            } else {
                ratio = newFeedRatio;
            }
            feedRatioTextField.setValue(new Double(ratio));
        }
    }//GEN-LAST:event_newInitialFeedRatio


  // Variables declaration - do not modify//GEN-BEGIN:variables
  private javax.swing.JLabel conversionLabel;
  private javax.swing.JLabel enthalpyLabel;
  private javax.swing.JLabel entropyLabel;
  private javax.swing.JLabel equilConstantLabel;
  private javax.swing.JButton equilibrateButton;
  private javax.swing.JFormattedTextField feedRatioTextField;
  private javax.swing.JLabel freeEnergyLabel;
  private javax.swing.JLabel jLabel11;
  private javax.swing.JLabel jLabel12;
  private javax.swing.JLabel jLabel13;
  private javax.swing.JLabel jLabel14;
  private javax.swing.JLabel jLabel15;
  private javax.swing.JLabel jLabel16;
  private javax.swing.JLabel jLabel18;
  private javax.swing.JMenuItem jMenuItem5;
  private javax.swing.JPanel jPanel1;
  private javax.swing.JPanel jPanel2;
  private javax.swing.JPanel jPanel3;
  private javax.swing.JPanel mainPanel;
  private javax.swing.JMenuBar menuBar;
  private javax.swing.JLabel percentCOLabel;
  private javax.swing.JLabel percentH2Label;
  private javax.swing.JLabel percentMeOHLabel;
  private javax.swing.JFormattedTextField pressureTextField;
  private javax.swing.JFormattedTextField temperatureTextField;
  // End of variables declaration//GEN-END:variables

    private JDialog aboutBox;
    private JDialog userGuide;
    private double T, P, ratio, n0CO, n0H2;
    private Shomate H2, CO, MeOH;
    private double percentH2, percentCO, percentMeOH;
}
