/*
 * StaticReactorsView.java
 */
package staticreactors;

import java.io.IOException;
import numeth.NMResult;
import org.jdesktop.application.Action;
import org.jdesktop.application.SingleFrameApplication;
import org.jdesktop.application.FrameView;
import org.jdesktop.application.Task;
import java.io.File;
import java.io.FileWriter;
import java.text.DecimalFormat;
import javax.swing.JDialog;
import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JOptionPane;
import numeth.NMEulerEqns;
import numeth.NMFixedEulerListener;
import numeth.NMFixedEulerSolver;

/**
 * The application's main frame.
 */
public class StaticReactorsView extends FrameView {
  // A few output formats

  DecimalFormat threeSigFigs = new DecimalFormat("0.00E00");
  DecimalFormat notScientific0 = new DecimalFormat("##0");
  DecimalFormat notScientific2 = new DecimalFormat("##0.00");
  DecimalFormat notScientific4 = new DecimalFormat("##0.0000");

  public StaticReactorsView(SingleFrameApplication app) {
    super(app);

    initComponents();

    //                Settings for Customizing the Simulator
    // ************************************************************************
    // Reactor volume in L
    vReactor = 100.0;
    // Reaction order with respect to A (0, 1 or 2)
    alpha = 1;
    // Reaction order with respect to B (0, 1 or 2)
    beta = 1;
    // Pre-exponential factor in units of mol, min and L
    k0 = 7.32e7;
    // Activation energy in J/mol
    actE = 50000.0;
    // Heat of reaction in J/mol
    deltaH = -60000.0;
    // Fluid density in kg/L
    rho = 1.0;
    // Fluid heat capacity in J/kg
    heatCapacity = 4186.0;
    // Reactor type label (used in output header)
    reactorType = "Isothermal PFR";
    // Uncertainty in concentration and temperature (used to add random error)
    concUncertainty = 0.075;
    temperatureUncertainty = 2.0;
    // Allowed ranges for inlet settings
    minimumInletT = 270.0;
    maximumInletT = 360.0;
    minimumInletFlow = 1.0;
    maximumInletFlow = 1000.0;
    minimumInletConcA = 0.01;
    maximumInletConcA = 5.0;
    minimumInletConcB = 0.01;
    maximumInletConcB = 5.0;
    minimumInletConcY = 0.0;
    maximumInletConcY = 5.0;
    minimumInletConcZ = 0.0;
    maximumInletConcZ = 5.0;
    // Default values for inlet settings
    defaultInletTemperature = 298.15;
    defaultInletFlowRate = 5.0;
    defaultInletConcA = 0.5;
    defaultInletConcB = 0.5;
    defaultInletConcY = 0.0;
    defaultInletConcZ = 0.0;
    // ************************************************************************

    // Ideal gas constant in J/mol/K
    gasConst = 8.3144;

    // Set the current inlet settings to the default values
    inletTemperature = defaultInletTemperature;
    inletFlowRate = defaultInletFlowRate;
    inletConcA = defaultInletConcA;
    inletConcB = defaultInletConcB;
    inletConcY = defaultInletConcY;
    inletConcZ = defaultInletConcZ;

    // Set values for outlet; shouldn't ever be used
    outletConcY = defaultInletConcY;
    outletTemperature = defaultInletTemperature;

    // Display the inlet settings
    temperatureFormattedTextField.setValue(inletTemperature);
    flowRateFormattedTextField.setValue(inletFlowRate);
    concAFormattedTextField.setValue(inletConcA);
    concBFormattedTextField.setValue(inletConcB);
    concYFormattedTextField.setValue(inletConcY);
    concZFormattedTextField.setValue(inletConcZ);

    // Set the "Tool Tips" for the inlet settings
    String text = notScientific0.format(minimumInletT) + " to ";
    text += notScientific0.format(maximumInletT);
    temperatureFormattedTextField.setToolTipText(text);
    text = notScientific2.format(minimumInletFlow) + " to ";
    text += notScientific2.format(maximumInletFlow);
    flowRateFormattedTextField.setToolTipText(text);
    text = notScientific2.format(minimumInletConcA) + " to ";
    text += notScientific2.format(maximumInletConcA);
    concAFormattedTextField.setToolTipText(text);
    text = notScientific2.format(minimumInletConcB) + " to ";
    text += notScientific2.format(maximumInletConcB);
    concBFormattedTextField.setToolTipText(text);
    text = notScientific2.format(minimumInletConcY) + " to ";
    text += notScientific2.format(maximumInletConcY);
    concYFormattedTextField.setToolTipText(text);
    text = notScientific2.format(minimumInletConcZ) + " to ";
    text += notScientific2.format(maximumInletConcZ);
    concZFormattedTextField.setToolTipText(text);

    // Enable all buttons except "Add Experiment to Data Set"
    recordDataButton.setEnabled(false);

    // Set the outlet condition to unsteady
    outletConcLabel.setText(unsteady1);
    outletConcValueLabel.setText(space);
    outletTemperatureLabel.setText(unsteady2);
    outletTemperatureValueLabel.setText(space);

    // Write a header to the output pane
    header = reactorType + " Reactor Data for the reaction A + B --> Y + Z\n";
    header += "Total Reactor Volume: " + notScientific0.format(vReactor);
    header += " L\n";
    header += "Inlet\tInlet\tInlet\tInlet\tInlet\tInlet\tOutlet\tOutlet\n";
    header += "T (K)\tFlow (L/min)\tConc. A (M)\tConc. B (M)\tConc. Y (M)\t";
    header += "Conc. Z (M)\tT (K)\tConc. Y (M)\n";
    outputTextArea.setText(header);
  }

  @Action
  public void showAboutBox() {
    if (aboutBox == null) {
      JFrame mainFrame = StaticReactorsApp.getApplication().getMainFrame();
      aboutBox = new StaticReactorsAboutBox(mainFrame);
      aboutBox.setLocationRelativeTo(mainFrame);
    }
    StaticReactorsApp.getApplication().show(aboutBox);
  }

  /** This method is called from within the constructor to
   * initialize the form.
   * WARNING: Do NOT modify this code. The content of this method is
   * always regenerated by the Form Editor.
   */
  @SuppressWarnings("unchecked")
  // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
  private void initComponents() {

    mainPanel = new javax.swing.JPanel();
    inletSettingsPanel = new javax.swing.JPanel();
    inletTemperatureLabel = new javax.swing.JLabel();
    flowRateLabel = new javax.swing.JLabel();
    temperatureFormattedTextField = new javax.swing.JFormattedTextField();
    flowRateFormattedTextField = new javax.swing.JFormattedTextField();
    concentrationLabel = new javax.swing.JLabel();
    concALabel = new javax.swing.JLabel();
    concAFormattedTextField = new javax.swing.JFormattedTextField();
    concBLabel = new javax.swing.JLabel();
    concBFormattedTextField = new javax.swing.JFormattedTextField();
    concYLabel = new javax.swing.JLabel();
    concYFormattedTextField = new javax.swing.JFormattedTextField();
    concZLabel = new javax.swing.JLabel();
    concZFormattedTextField = new javax.swing.JFormattedTextField();
    diagramLabel = new javax.swing.JLabel();
    startExptButton = new javax.swing.JButton();
    outletPanel = new javax.swing.JPanel();
    outletConcLabel = new javax.swing.JLabel();
    outletConcValueLabel = new javax.swing.JLabel();
    outletTemperatureLabel = new javax.swing.JLabel();
    outletTemperatureValueLabel = new javax.swing.JLabel();
    recordDataButton = new javax.swing.JButton();
    jScrollPane1 = new javax.swing.JScrollPane();
    outputTextArea = new javax.swing.JTextArea();
    saveButton = new javax.swing.JButton();
    clearButton = new javax.swing.JButton();
    dataSetLabel = new javax.swing.JLabel();
    menuBar = new javax.swing.JMenuBar();
    javax.swing.JMenu fileMenu = new javax.swing.JMenu();
    saveMenuItem = new javax.swing.JMenuItem();
    javax.swing.JMenuItem exitMenuItem = new javax.swing.JMenuItem();
    javax.swing.JMenu helpMenu = new javax.swing.JMenu();
    javax.swing.JMenuItem aboutMenuItem = new javax.swing.JMenuItem();
    userGuideMenuItem = new javax.swing.JMenuItem();

    mainPanel.setName("mainPanel"); // NOI18N
    mainPanel.setPreferredSize(new java.awt.Dimension(900, 600));

    org.jdesktop.application.ResourceMap resourceMap = org.jdesktop.application.Application.getInstance(staticreactors.StaticReactorsApp.class).getContext().getResourceMap(StaticReactorsView.class);
    inletSettingsPanel.setBorder(javax.swing.BorderFactory.createTitledBorder(resourceMap.getString("inletSettingsPanel.border.title"))); // NOI18N
    inletSettingsPanel.setName("inletSettingsPanel"); // NOI18N
    inletSettingsPanel.setPreferredSize(new java.awt.Dimension(300, 300));

    inletTemperatureLabel.setText(resourceMap.getString("inletTemperatureLabel.text")); // NOI18N
    inletTemperatureLabel.setName("inletTemperatureLabel"); // NOI18N

    flowRateLabel.setText(resourceMap.getString("flowRateLabel.text")); // NOI18N
    flowRateLabel.setName("flowRateLabel"); // NOI18N

    temperatureFormattedTextField.setFormatterFactory(new javax.swing.text.DefaultFormatterFactory(new javax.swing.text.NumberFormatter(new java.text.DecimalFormat("#,##0"))));
    temperatureFormattedTextField.setHorizontalAlignment(javax.swing.JTextField.CENTER);
    temperatureFormattedTextField.setText(resourceMap.getString("temperatureFormattedTextField.text")); // NOI18N
    temperatureFormattedTextField.setName("temperatureFormattedTextField"); // NOI18N
    temperatureFormattedTextField.addFocusListener(new java.awt.event.FocusAdapter() {
      public void focusGained(java.awt.event.FocusEvent evt) {
        temperatureFormattedTextFieldFocusGained(evt);
      }
    });
    temperatureFormattedTextField.addPropertyChangeListener(new java.beans.PropertyChangeListener() {
      public void propertyChange(java.beans.PropertyChangeEvent evt) {
        newInletTemperature(evt);
      }
    });

    flowRateFormattedTextField.setFormatterFactory(new javax.swing.text.DefaultFormatterFactory(new javax.swing.text.NumberFormatter(new java.text.DecimalFormat("#,##0.00"))));
    flowRateFormattedTextField.setHorizontalAlignment(javax.swing.JTextField.CENTER);
    flowRateFormattedTextField.setText(resourceMap.getString("flowRateFormattedTextField.text")); // NOI18N
    flowRateFormattedTextField.setName("flowRateFormattedTextField"); // NOI18N
    flowRateFormattedTextField.addFocusListener(new java.awt.event.FocusAdapter() {
      public void focusGained(java.awt.event.FocusEvent evt) {
        flowRateFormattedTextFieldFocusGained(evt);
      }
    });
    flowRateFormattedTextField.addPropertyChangeListener(new java.beans.PropertyChangeListener() {
      public void propertyChange(java.beans.PropertyChangeEvent evt) {
        newInletFlowRate(evt);
      }
    });

    concentrationLabel.setHorizontalAlignment(javax.swing.SwingConstants.CENTER);
    concentrationLabel.setText(resourceMap.getString("concentrationLabel.text")); // NOI18N
    concentrationLabel.setName("concentrationLabel"); // NOI18N

    concALabel.setText(resourceMap.getString("concALabel.text")); // NOI18N
    concALabel.setName("concALabel"); // NOI18N

    concAFormattedTextField.setFormatterFactory(new javax.swing.text.DefaultFormatterFactory(new javax.swing.text.NumberFormatter(new java.text.DecimalFormat("#,##0.00"))));
    concAFormattedTextField.setHorizontalAlignment(javax.swing.JTextField.CENTER);
    concAFormattedTextField.setText(resourceMap.getString("concAFormattedTextField.text")); // NOI18N
    concAFormattedTextField.setName("concAFormattedTextField"); // NOI18N
    concAFormattedTextField.addFocusListener(new java.awt.event.FocusAdapter() {
      public void focusGained(java.awt.event.FocusEvent evt) {
        concAFormattedTextFieldFocusGained(evt);
      }
    });
    concAFormattedTextField.addPropertyChangeListener(new java.beans.PropertyChangeListener() {
      public void propertyChange(java.beans.PropertyChangeEvent evt) {
        newInletConcA(evt);
      }
    });

    concBLabel.setText(resourceMap.getString("concBLabel.text")); // NOI18N
    concBLabel.setName("concBLabel"); // NOI18N

    concBFormattedTextField.setFormatterFactory(new javax.swing.text.DefaultFormatterFactory(new javax.swing.text.NumberFormatter(new java.text.DecimalFormat("#,##0.00"))));
    concBFormattedTextField.setHorizontalAlignment(javax.swing.JTextField.CENTER);
    concBFormattedTextField.setText(resourceMap.getString("concBFormattedTextField.text")); // NOI18N
    concBFormattedTextField.setName("concBFormattedTextField"); // NOI18N
    concBFormattedTextField.addFocusListener(new java.awt.event.FocusAdapter() {
      public void focusGained(java.awt.event.FocusEvent evt) {
        concBFormattedTextFieldFocusGained(evt);
      }
    });
    concBFormattedTextField.addPropertyChangeListener(new java.beans.PropertyChangeListener() {
      public void propertyChange(java.beans.PropertyChangeEvent evt) {
        newInletConcB(evt);
      }
    });

    concYLabel.setText(resourceMap.getString("concYLabel.text")); // NOI18N
    concYLabel.setName("concYLabel"); // NOI18N

    concYFormattedTextField.setFormatterFactory(new javax.swing.text.DefaultFormatterFactory(new javax.swing.text.NumberFormatter(new java.text.DecimalFormat("#,##0.00"))));
    concYFormattedTextField.setHorizontalAlignment(javax.swing.JTextField.CENTER);
    concYFormattedTextField.setText(resourceMap.getString("concYFormattedTextField.text")); // NOI18N
    concYFormattedTextField.setName("concYFormattedTextField"); // NOI18N
    concYFormattedTextField.addFocusListener(new java.awt.event.FocusAdapter() {
      public void focusGained(java.awt.event.FocusEvent evt) {
        concYFormattedTextFieldFocusGained(evt);
      }
    });
    concYFormattedTextField.addPropertyChangeListener(new java.beans.PropertyChangeListener() {
      public void propertyChange(java.beans.PropertyChangeEvent evt) {
        newInletConcY(evt);
      }
    });

    concZLabel.setText(resourceMap.getString("concZLabel.text")); // NOI18N
    concZLabel.setName("concZLabel"); // NOI18N

    concZFormattedTextField.setFormatterFactory(new javax.swing.text.DefaultFormatterFactory(new javax.swing.text.NumberFormatter(new java.text.DecimalFormat("#,##0.00"))));
    concZFormattedTextField.setHorizontalAlignment(javax.swing.JTextField.CENTER);
    concZFormattedTextField.setText(resourceMap.getString("concZFormattedTextField.text")); // NOI18N
    concZFormattedTextField.setName("concZFormattedTextField"); // NOI18N
    concZFormattedTextField.addFocusListener(new java.awt.event.FocusAdapter() {
      public void focusGained(java.awt.event.FocusEvent evt) {
        concZFormattedTextFieldFocusGained(evt);
      }
    });
    concZFormattedTextField.addPropertyChangeListener(new java.beans.PropertyChangeListener() {
      public void propertyChange(java.beans.PropertyChangeEvent evt) {
        newInletConcZ(evt);
      }
    });

    javax.swing.GroupLayout inletSettingsPanelLayout = new javax.swing.GroupLayout(inletSettingsPanel);
    inletSettingsPanel.setLayout(inletSettingsPanelLayout);
    inletSettingsPanelLayout.setHorizontalGroup(
      inletSettingsPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
      .addGroup(inletSettingsPanelLayout.createSequentialGroup()
        .addContainerGap()
        .addGroup(inletSettingsPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
          .addGroup(inletSettingsPanelLayout.createSequentialGroup()
            .addGroup(inletSettingsPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.TRAILING, false)
              .addComponent(flowRateLabel, javax.swing.GroupLayout.Alignment.LEADING, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
              .addComponent(inletTemperatureLabel, javax.swing.GroupLayout.Alignment.LEADING, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE))
            .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
            .addGroup(inletSettingsPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
              .addComponent(flowRateFormattedTextField, javax.swing.GroupLayout.DEFAULT_SIZE, 98, Short.MAX_VALUE)
              .addComponent(temperatureFormattedTextField, javax.swing.GroupLayout.DEFAULT_SIZE, 98, Short.MAX_VALUE)))
          .addComponent(concentrationLabel, javax.swing.GroupLayout.DEFAULT_SIZE, 214, Short.MAX_VALUE)
          .addGroup(inletSettingsPanelLayout.createSequentialGroup()
            .addComponent(concALabel, javax.swing.GroupLayout.DEFAULT_SIZE, 87, Short.MAX_VALUE)
            .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
            .addComponent(concAFormattedTextField, javax.swing.GroupLayout.DEFAULT_SIZE, 121, Short.MAX_VALUE))
          .addGroup(inletSettingsPanelLayout.createSequentialGroup()
            .addComponent(concBLabel, javax.swing.GroupLayout.DEFAULT_SIZE, 87, Short.MAX_VALUE)
            .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
            .addComponent(concBFormattedTextField, javax.swing.GroupLayout.DEFAULT_SIZE, 121, Short.MAX_VALUE))
          .addGroup(javax.swing.GroupLayout.Alignment.TRAILING, inletSettingsPanelLayout.createSequentialGroup()
            .addComponent(concYLabel, javax.swing.GroupLayout.DEFAULT_SIZE, 87, Short.MAX_VALUE)
            .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
            .addComponent(concYFormattedTextField, javax.swing.GroupLayout.DEFAULT_SIZE, 121, Short.MAX_VALUE))
          .addGroup(inletSettingsPanelLayout.createSequentialGroup()
            .addComponent(concZLabel, javax.swing.GroupLayout.DEFAULT_SIZE, 87, Short.MAX_VALUE)
            .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
            .addComponent(concZFormattedTextField, javax.swing.GroupLayout.DEFAULT_SIZE, 121, Short.MAX_VALUE)))
        .addContainerGap())
    );
    inletSettingsPanelLayout.setVerticalGroup(
      inletSettingsPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
      .addGroup(inletSettingsPanelLayout.createSequentialGroup()
        .addContainerGap()
        .addGroup(inletSettingsPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
          .addComponent(inletTemperatureLabel)
          .addComponent(temperatureFormattedTextField, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.UNRELATED)
        .addGroup(inletSettingsPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
          .addComponent(flowRateLabel)
          .addComponent(flowRateFormattedTextField, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
        .addComponent(concentrationLabel)
        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
        .addGroup(inletSettingsPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
          .addComponent(concALabel)
          .addComponent(concAFormattedTextField, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
        .addGroup(inletSettingsPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
          .addComponent(concBLabel)
          .addComponent(concBFormattedTextField, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
        .addGroup(inletSettingsPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
          .addComponent(concYLabel)
          .addComponent(concYFormattedTextField, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
        .addGroup(inletSettingsPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
          .addComponent(concZLabel)
          .addComponent(concZFormattedTextField, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
        .addGap(17, 17, 17))
    );

    diagramLabel.setHorizontalAlignment(javax.swing.SwingConstants.CENTER);
    diagramLabel.setIcon(resourceMap.getIcon("diagramLabel.icon")); // NOI18N
    diagramLabel.setText(resourceMap.getString("diagramLabel.text")); // NOI18N
    diagramLabel.setName("diagramLabel"); // NOI18N

    javax.swing.ActionMap actionMap = org.jdesktop.application.Application.getInstance(staticreactors.StaticReactorsApp.class).getContext().getActionMap(StaticReactorsView.class, this);
    startExptButton.setAction(actionMap.get("startExperiment")); // NOI18N
    startExptButton.setText(resourceMap.getString("startExptButton.text")); // NOI18N
    startExptButton.setName("startExptButton"); // NOI18N

    outletPanel.setBorder(javax.swing.BorderFactory.createTitledBorder(resourceMap.getString("outletPanel.border.title"))); // NOI18N
    outletPanel.setName("outletPanel"); // NOI18N

    outletConcLabel.setHorizontalAlignment(javax.swing.SwingConstants.CENTER);
    outletConcLabel.setText(resourceMap.getString("outletConcLabel.text")); // NOI18N
    outletConcLabel.setName("outletConcLabel"); // NOI18N

    outletConcValueLabel.setHorizontalAlignment(javax.swing.SwingConstants.CENTER);
    outletConcValueLabel.setText(resourceMap.getString("outletConcValueLabel.text")); // NOI18N
    outletConcValueLabel.setName("outletConcValueLabel"); // NOI18N

    outletTemperatureLabel.setHorizontalAlignment(javax.swing.SwingConstants.CENTER);
    outletTemperatureLabel.setText(resourceMap.getString("outletTemperatureLabel.text")); // NOI18N
    outletTemperatureLabel.setName("outletTemperatureLabel"); // NOI18N

    outletTemperatureValueLabel.setHorizontalAlignment(javax.swing.SwingConstants.CENTER);
    outletTemperatureValueLabel.setText(resourceMap.getString("outletTemperatureValueLabel.text")); // NOI18N
    outletTemperatureValueLabel.setName("outletTemperatureValueLabel"); // NOI18N

    javax.swing.GroupLayout outletPanelLayout = new javax.swing.GroupLayout(outletPanel);
    outletPanel.setLayout(outletPanelLayout);
    outletPanelLayout.setHorizontalGroup(
      outletPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
      .addGroup(outletPanelLayout.createSequentialGroup()
        .addContainerGap()
        .addGroup(outletPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
          .addComponent(outletTemperatureLabel, javax.swing.GroupLayout.DEFAULT_SIZE, 230, Short.MAX_VALUE)
          .addGroup(outletPanelLayout.createSequentialGroup()
            .addGap(6, 6, 6)
            .addComponent(outletConcValueLabel, javax.swing.GroupLayout.DEFAULT_SIZE, 224, Short.MAX_VALUE))
          .addComponent(outletConcLabel, javax.swing.GroupLayout.DEFAULT_SIZE, 230, Short.MAX_VALUE)
          .addComponent(outletTemperatureValueLabel, javax.swing.GroupLayout.DEFAULT_SIZE, 230, Short.MAX_VALUE))
        .addContainerGap())
    );
    outletPanelLayout.setVerticalGroup(
      outletPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
      .addGroup(outletPanelLayout.createSequentialGroup()
        .addGap(32, 32, 32)
        .addComponent(outletConcLabel)
        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
        .addComponent(outletConcValueLabel)
        .addGap(30, 30, 30)
        .addComponent(outletTemperatureLabel)
        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
        .addComponent(outletTemperatureValueLabel)
        .addContainerGap(39, Short.MAX_VALUE))
    );

    recordDataButton.setAction(actionMap.get("addExperimentToDataSet")); // NOI18N
    recordDataButton.setText(resourceMap.getString("recordDataButton.text")); // NOI18N
    recordDataButton.setName("recordDataButton"); // NOI18N

    jScrollPane1.setName("jScrollPane1"); // NOI18N

    outputTextArea.setColumns(20);
    outputTextArea.setFont(resourceMap.getFont("outputTextArea.font")); // NOI18N
    outputTextArea.setRows(5);
    outputTextArea.setName("outputTextArea"); // NOI18N
    jScrollPane1.setViewportView(outputTextArea);

    saveButton.setAction(actionMap.get("saveDataSet")); // NOI18N
    saveButton.setText(resourceMap.getString("saveButton.text")); // NOI18N
    saveButton.setName("saveButton"); // NOI18N

    clearButton.setAction(actionMap.get("clearDataSet")); // NOI18N
    clearButton.setText(resourceMap.getString("clearButton.text")); // NOI18N
    clearButton.setName("clearButton"); // NOI18N

    dataSetLabel.setFont(resourceMap.getFont("dataSetLabel.font")); // NOI18N
    dataSetLabel.setHorizontalAlignment(javax.swing.SwingConstants.CENTER);
    dataSetLabel.setText(resourceMap.getString("dataSetLabel.text")); // NOI18N
    dataSetLabel.setName("dataSetLabel"); // NOI18N

    javax.swing.GroupLayout mainPanelLayout = new javax.swing.GroupLayout(mainPanel);
    mainPanel.setLayout(mainPanelLayout);
    mainPanelLayout.setHorizontalGroup(
      mainPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
      .addGroup(mainPanelLayout.createSequentialGroup()
        .addGap(113, 113, 113)
        .addComponent(saveButton, javax.swing.GroupLayout.PREFERRED_SIZE, 188, javax.swing.GroupLayout.PREFERRED_SIZE)
        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED, 284, Short.MAX_VALUE)
        .addComponent(clearButton, javax.swing.GroupLayout.PREFERRED_SIZE, 188, javax.swing.GroupLayout.PREFERRED_SIZE)
        .addGap(131, 131, 131))
      .addGroup(mainPanelLayout.createSequentialGroup()
        .addContainerGap()
        .addGroup(mainPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
          .addComponent(jScrollPane1, javax.swing.GroupLayout.Alignment.TRAILING, javax.swing.GroupLayout.DEFAULT_SIZE, 892, Short.MAX_VALUE)
          .addGroup(mainPanelLayout.createSequentialGroup()
            .addComponent(inletSettingsPanel, javax.swing.GroupLayout.PREFERRED_SIZE, 238, javax.swing.GroupLayout.PREFERRED_SIZE)
            .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
            .addComponent(diagramLabel, javax.swing.GroupLayout.DEFAULT_SIZE, 388, Short.MAX_VALUE)
            .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
            .addGroup(mainPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.TRAILING, false)
              .addComponent(recordDataButton, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
              .addComponent(startExptButton, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
              .addComponent(outletPanel, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE))))
        .addContainerGap())
      .addComponent(dataSetLabel, javax.swing.GroupLayout.DEFAULT_SIZE, 904, Short.MAX_VALUE)
    );
    mainPanelLayout.setVerticalGroup(
      mainPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
      .addGroup(mainPanelLayout.createSequentialGroup()
        .addContainerGap()
        .addGroup(mainPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
          .addComponent(diagramLabel, javax.swing.GroupLayout.PREFERRED_SIZE, 277, Short.MAX_VALUE)
          .addComponent(inletSettingsPanel, javax.swing.GroupLayout.Alignment.TRAILING, javax.swing.GroupLayout.DEFAULT_SIZE, 277, Short.MAX_VALUE)
          .addGroup(mainPanelLayout.createSequentialGroup()
            .addGap(18, 18, 18)
            .addComponent(startExptButton)
            .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
            .addComponent(outletPanel, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
            .addGap(13, 13, 13)
            .addComponent(recordDataButton)))
        .addGap(18, 18, 18)
        .addComponent(dataSetLabel)
        .addGap(16, 16, 16)
        .addComponent(jScrollPane1, javax.swing.GroupLayout.PREFERRED_SIZE, 242, javax.swing.GroupLayout.PREFERRED_SIZE)
        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.UNRELATED)
        .addGroup(mainPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
          .addComponent(saveButton)
          .addComponent(clearButton))
        .addGap(163, 163, 163))
    );

    menuBar.setName("menuBar"); // NOI18N

    fileMenu.setText(resourceMap.getString("fileMenu.text")); // NOI18N
    fileMenu.setName("fileMenu"); // NOI18N

    saveMenuItem.setAction(actionMap.get("saveDataSet")); // NOI18N
    saveMenuItem.setText(resourceMap.getString("saveMenuItem.text")); // NOI18N
    saveMenuItem.setName("saveMenuItem"); // NOI18N
    fileMenu.add(saveMenuItem);

    exitMenuItem.setAction(actionMap.get("quit")); // NOI18N
    exitMenuItem.setName("exitMenuItem"); // NOI18N
    fileMenu.add(exitMenuItem);

    menuBar.add(fileMenu);

    helpMenu.setText(resourceMap.getString("helpMenu.text")); // NOI18N
    helpMenu.setName("helpMenu"); // NOI18N

    aboutMenuItem.setAction(actionMap.get("showAboutBox")); // NOI18N
    aboutMenuItem.setName("aboutMenuItem"); // NOI18N
    helpMenu.add(aboutMenuItem);

    userGuideMenuItem.setAction(actionMap.get("showUserGuide")); // NOI18N
    userGuideMenuItem.setText(resourceMap.getString("userGuideMenuItem.text")); // NOI18N
    userGuideMenuItem.setName("userGuideMenuItem"); // NOI18N
    helpMenu.add(userGuideMenuItem);

    menuBar.add(helpMenu);

    setComponent(mainPanel);
    setMenuBar(menuBar);
  }// </editor-fold>//GEN-END:initComponents

    private void newInletTemperature(java.beans.PropertyChangeEvent evt) {//GEN-FIRST:event_newInletTemperature
      // Disable the "Add Experiment to Data Set Button" if it is enabled
      enableAddExpt(false);
      // Set the outlet condition to unsteady
      setOutletUnsteady();
      // see whether the value has really changed
      double newValue = ((Number) temperatureFormattedTextField.getValue()).doubleValue();
      if (newValue != inletTemperature) {
        // the value has changed ; make sure it is in range
        if (newValue < minimumInletT) {
          inletTemperature = minimumInletT;
        } else if (newValue > maximumInletT) {
          inletTemperature = maximumInletT;
        } else {
          inletTemperature = newValue;
        }
        temperatureFormattedTextField.setValue(new Double(inletTemperature));
      }
    }//GEN-LAST:event_newInletTemperature

    private void newInletFlowRate(java.beans.PropertyChangeEvent evt) {//GEN-FIRST:event_newInletFlowRate
      // Disable the "Add Experiment to Data Set Button" if it is enabled
      enableAddExpt(false);
      // Set the outlet condition to unsteady
      setOutletUnsteady();
      // see whether the value has really changed
      double newValue = ((Number) flowRateFormattedTextField.getValue()).doubleValue();
      if (newValue != inletFlowRate) {
        // the value has changed ; make sure it is in range
        if (newValue < minimumInletFlow) {
          inletFlowRate = minimumInletFlow;
        } else if (newValue > maximumInletFlow) {
          inletFlowRate = maximumInletFlow;
        } else {
          inletFlowRate = newValue;
        }
        flowRateFormattedTextField.setValue(new Double(inletFlowRate));
      }
    }//GEN-LAST:event_newInletFlowRate

    private void newInletConcA(java.beans.PropertyChangeEvent evt) {//GEN-FIRST:event_newInletConcA
      // Disable the "Add Experiment to Data Set Button" if it is enabled
      enableAddExpt(false);
      // Set the outlet condition to unsteady
      setOutletUnsteady();
      // see whether the value has really changed
      double newValue = ((Number) concAFormattedTextField.getValue()).doubleValue();
      if (newValue != inletConcA) {
        // the value has changed ; make sure it is in range
        if (newValue < minimumInletConcA) {
          inletConcA = minimumInletConcA;
        } else if (newValue > maximumInletConcA) {
          inletConcA = maximumInletConcA;
        } else {
          inletConcA = newValue;
        }
        concAFormattedTextField.setValue(new Double(inletConcA));
      }
    }//GEN-LAST:event_newInletConcA

    private void newInletConcB(java.beans.PropertyChangeEvent evt) {//GEN-FIRST:event_newInletConcB
      // Disable the "Add Experiment to Data Set Button" if it is enabled
      enableAddExpt(false);
      // Set the outlet condition to unsteady
      setOutletUnsteady();
      // see whether the value has really changed
      double newValue = ((Number) concBFormattedTextField.getValue()).doubleValue();
      if (newValue != inletConcB) {
        // the value has changed ; make sure it is in range
        if (newValue < minimumInletConcB) {
          inletConcB = minimumInletConcB;
        } else if (newValue > maximumInletConcB) {
          inletConcB = maximumInletConcB;
        } else {
          inletConcB = newValue;
        }
        concBFormattedTextField.setValue(new Double(inletConcB));
      }
    }//GEN-LAST:event_newInletConcB

    private void newInletConcY(java.beans.PropertyChangeEvent evt) {//GEN-FIRST:event_newInletConcY
      // Disable the "Add Experiment to Data Set Button" if it is enabled
      enableAddExpt(false);
      // Set the outlet condition to unsteady
      setOutletUnsteady();
      // see whether the value has really changed
      double newValue = ((Number) concYFormattedTextField.getValue()).doubleValue();
      if (newValue != inletConcY) {
        // the value has changed ; make sure it is in range
        if (newValue < minimumInletConcY) {
          inletConcY = minimumInletConcY;
        } else if (newValue > maximumInletConcY) {
          inletConcY = maximumInletConcY;
        } else {
          inletConcY = newValue;
        }
        concYFormattedTextField.setValue(new Double(inletConcY));
      }
    }//GEN-LAST:event_newInletConcY

    private void newInletConcZ(java.beans.PropertyChangeEvent evt) {//GEN-FIRST:event_newInletConcZ
      // Disable the "Add Experiment to Data Set Button" if it is enabled
      enableAddExpt(false);
      // Set the outlet condition to unsteady
      setOutletUnsteady();
      // see whether the value has really changed
      double newValue = ((Number) concZFormattedTextField.getValue()).doubleValue();
      if (newValue != inletConcZ) {
        // the value has changed ; make sure it is in range
        if (newValue < minimumInletConcZ) {
          inletConcA = minimumInletConcA;
        } else if (newValue > maximumInletConcZ) {
          inletConcZ = maximumInletConcZ;
        } else {
          inletConcZ = newValue;
        }
        concZFormattedTextField.setValue(new Double(inletConcZ));
      }
    }//GEN-LAST:event_newInletConcZ

    private void temperatureFormattedTextFieldFocusGained(java.awt.event.FocusEvent evt) {//GEN-FIRST:event_temperatureFormattedTextFieldFocusGained
      enableAddExpt(false);
    }//GEN-LAST:event_temperatureFormattedTextFieldFocusGained

    private void flowRateFormattedTextFieldFocusGained(java.awt.event.FocusEvent evt) {//GEN-FIRST:event_flowRateFormattedTextFieldFocusGained
      enableAddExpt(false);
    }//GEN-LAST:event_flowRateFormattedTextFieldFocusGained

    private void concAFormattedTextFieldFocusGained(java.awt.event.FocusEvent evt) {//GEN-FIRST:event_concAFormattedTextFieldFocusGained
      enableAddExpt(false);
    }//GEN-LAST:event_concAFormattedTextFieldFocusGained

    private void concBFormattedTextFieldFocusGained(java.awt.event.FocusEvent evt) {//GEN-FIRST:event_concBFormattedTextFieldFocusGained
      enableAddExpt(false);
    }//GEN-LAST:event_concBFormattedTextFieldFocusGained

    private void concYFormattedTextFieldFocusGained(java.awt.event.FocusEvent evt) {//GEN-FIRST:event_concYFormattedTextFieldFocusGained
      enableAddExpt(false);
    }//GEN-LAST:event_concYFormattedTextFieldFocusGained

    private void concZFormattedTextFieldFocusGained(java.awt.event.FocusEvent evt) {//GEN-FIRST:event_concZFormattedTextFieldFocusGained
      enableAddExpt(false);
    }//GEN-LAST:event_concZFormattedTextFieldFocusGained
  StartExperimentTask exptTask;

  @Action
  public void showUserGuide() {
    if (userGuide == null) {
      JFrame mainFrame = StaticReactorsApp.getApplication().getMainFrame();
      userGuide = new StaticReactorsUserGuide(mainFrame);
      userGuide.setLocationRelativeTo(mainFrame);
    }
    StaticReactorsApp.getApplication().show(userGuide);
  }

  @Action
  public Task startExperiment() {
    return exptTask = new StartExperimentTask(getApplication());
  }

  private class StartExperimentTask extends org.jdesktop.application.Task<Object, Void>
          implements NMEulerEqns, NMFixedEulerListener {
    // Declare variables to hold EDT quantities used in the background thread

    double T0, VFR0, CA0, CB0, CY0, CZ0;
    // Declare constants that depend on inlet settings
    double tau, k;
    // Declare flag for convergence of the calculation
    boolean converged;

    StartExperimentTask(org.jdesktop.application.Application app) {
      // Runs on the EDT.  Copy GUI state that
      // doInBackground() depends on from parameters
      // to StartExperimentTask fields, here.
      super(app);

      // get inlet settings
      T0 = ((Number) temperatureFormattedTextField.getValue()).doubleValue();
      VFR0 = ((Number) flowRateFormattedTextField.getValue()).doubleValue();
      CA0 = ((Number) concAFormattedTextField.getValue()).doubleValue();
      CB0 = ((Number) concBFormattedTextField.getValue()).doubleValue();
      CY0 = ((Number) concYFormattedTextField.getValue()).doubleValue();
      CZ0 = ((Number) concZFormattedTextField.getValue()).doubleValue();
      // calculate constants that depend on the inlet settings
      tau = vReactor / VFR0;
      k = k0 * Math.exp(-actE / gasConst / T0);
      // disable the "Start Experiment" button
      enableStartExpt(false);
      // disable the "Add Experiment to Data Set Button" if it is enabled
      enableAddExpt(false);
      // disable input boxes for inlet settings
      temperatureFormattedTextField.setEnabled(false);
      flowRateFormattedTextField.setEnabled(false);
      concAFormattedTextField.setEnabled(false);
      concBFormattedTextField.setEnabled(false);
      concYFormattedTextField.setEnabled(false);
      concZFormattedTextField.setEnabled(false);
      // Set the outlet condition to unsteady
      setOutletUnsteady();
    }

    @Override
    protected Object doInBackground() {
      // Your Task's code here.  This method runs
      // on a background thread, so don't reference
      // the Swing GUI from here.

      // Solution not converged yet
      converged = false;

      // create and initialize an array for the concentration
      double[] CA = new double[1];
      CA[0] = CA0;

      // create the ode solver
      NMFixedEulerSolver theSolver = new NMFixedEulerSolver(this);
      theSolver.setTreatAsNonNegative(true);

      // integrate the ode's
      NMResult theResult = theSolver.integrate(0.0, tau, CA, 1000);

      // If calculation succeded, set converged to true
      if (theResult.getStatus() != 0) {
        return null;
      } else {
        converged = true;
        outletConcY = CY0 + CA0 - CA[0];
      }

      // The reactor is isothermal, so the outlet T equals the inlet T
      outletTemperature = T0;

      // Add random error to outlet values
      outletTemperature += temperatureUncertainty * ((Math.random() * 2.0) - 1.0);
      outletConcY += concUncertainty * ((Math.random() * 2.0) - 1.0);

      // Outlet values have been stored, so there isn't anything to return
      return null;
    }

    @Override
    protected void succeeded(Object result) {
      // Runs on the EDT.  Update the GUI based on
      // the result computed by doInBackground().
      // enable the input boxes for inlet settings
      temperatureFormattedTextField.setEnabled(true);
      flowRateFormattedTextField.setEnabled(true);
      concAFormattedTextField.setEnabled(true);
      concBFormattedTextField.setEnabled(true);
      concYFormattedTextField.setEnabled(true);
      concZFormattedTextField.setEnabled(true);
      // enable the "Start Experiment" button
      enableStartExpt(true);
      // write the steady-state outlet conditions using the newly stored values
      if (converged) {
        outletConcLabel.setText(steady1);
        outletConcValueLabel.setText(notScientific4.format(outletConcY));
        outletTemperatureLabel.setText(steady2);
        outletTemperatureValueLabel.setText(notScientific0.format(outletTemperature));
        // enable the "Add Experiment to Data Set Button"
        enableAddExpt(true);
      } else {
        outletConcLabel.setText(calcFailed1);
        outletConcValueLabel.setText(calcFailed2);
        outletTemperatureLabel.setText(calcFailed3);
        outletTemperatureValueLabel.setText(calcFailed4);
        enableAddExpt(false);
      }
    }

    public NMResult evalEulerEqns(double x, double[] y, double[] dydx) {
      // create a NMResult to return the result
      NMResult theResult = new NMResult();
      // evaluate the pfr design equation
      dydx[0] = -k;
      if (alpha != 0) {
        if (alpha == 2) {
          dydx[0] *= y[0] * y[0];
        } else {
          dydx[0] *= y[0];
        }
      }
      if (beta != 0) {
        if (beta == 2) {
          dydx[0] *= (CB0 - CA0 + y[0]) * (CB0 - CA0 + y[0]);
        } else {
          dydx[0] *= (CB0 - CA0 + y[0]);
        }
      }
      // values are stored, so return the result
      return theResult;
    }

    public NMResult evalEulerJac(double[] y, double[][] jac) {
      // create a NMResult to return the result
      NMResult theResult = new NMResult();
      double newTerm;
      // evaluate the jacobian of the pfr design equation
      jac[0][0] = 0.0;
      if (alpha != 0) {
        newTerm = -k * alpha;
        if (beta != 0) {
          if (beta == 2) {
            newTerm *= (CB0 - CA0 + y[0]) * (CB0 - CA0 + y[0]);
          } else {
            newTerm *= (CB0 - CA0 + y[0]);
          }
        }
        if (alpha == 2) {
          newTerm *= y[0];
        }
        jac[0][0] += newTerm;
      }
      if (beta != 0) {
        newTerm = -k * beta;
        if (beta == 2) {
          newTerm *= (CB0 - CA0 + y[0]);
        }
        if (alpha != 0) {
          if (alpha == 2) {
            newTerm *= y[0]*y[0];
          } else {
            newTerm *= y[0];
          }
        }
        jac[0][0] += newTerm;
      }
      // all the variables are saved; return the result
      return theResult;
    }

    public void call(int iteration, double indepVar, double[] depVar) {
      return;
    }
  }

  @Action
  public void clearDataSet() {
    // warn the user that data will be lost if not saved
    JFrame mainFrame = StaticReactorsApp.getApplication().getMainFrame();
    int n = JOptionPane.showConfirmDialog(
            mainFrame, "The data set will be lost if not saved, continue?",
            "Reset Warning", JOptionPane.WARNING_MESSAGE,
            JOptionPane.YES_NO_OPTION);
    // proceed if the user OK's it
    if (n == JOptionPane.YES_OPTION) {
      // if there is a calculation running, kill it
      if (exptTask != null) {
        exptTask.cancel(true);
      }

      // Set the current inlet settings to the default values
      inletTemperature = defaultInletTemperature;
      inletFlowRate = defaultInletFlowRate;
      inletConcA = defaultInletConcA;
      inletConcB = defaultInletConcB;
      inletConcY = defaultInletConcY;
      inletConcZ = defaultInletConcZ;

      // Display the inlet settings and enable the input fields
      temperatureFormattedTextField.setValue(inletTemperature);
      temperatureFormattedTextField.setEnabled(true);
      flowRateFormattedTextField.setValue(inletFlowRate);
      flowRateFormattedTextField.setEnabled(true);
      concAFormattedTextField.setValue(inletConcA);
      concAFormattedTextField.setEnabled(true);
      concBFormattedTextField.setValue(inletConcB);
      concBFormattedTextField.setEnabled(true);
      concYFormattedTextField.setValue(inletConcY);
      concYFormattedTextField.setEnabled(true);
      concZFormattedTextField.setValue(inletConcZ);
      concZFormattedTextField.setEnabled(true);

      // Enable all buttons except "Add Experiment to Data Set"
      enableStartExpt(true);
      enableAddExpt(false);

      // Set the outlet condition to unsteady
      setOutletUnsteady();

      // Write a header to the output pane
      header = reactorType + " Reactor Data for the reaction A + B --> Y + Z\n";
      header += "Total Reactor Volume: " + notScientific0.format(vReactor);
      header += " L\n";
      header += "Inlet\tInlet\tInlet\tInlet\tInlet\tInlet\tOutlet\tOutlet\n";
      header += "T (K)\tFlow (L/min)\tConc. A (M)\tConc. B (M)\tConc. Y (M)\t";
      header += "Conc. Z (M)\tT (K)\tConc. Y (M)\n";
      outputTextArea.setText(header);
    }
  }

  @Action
  public void addExperimentToDataSet() {
    String newData = notScientific0.format(inletTemperature) + "\t";
    newData += notScientific2.format(inletFlowRate) + "\t";
    newData += notScientific2.format(inletConcA) + "\t";
    newData += notScientific2.format(inletConcB) + "\t";
    newData += notScientific2.format(inletConcY) + "\t";
    newData += notScientific2.format(inletConcZ) + "\t";
    newData += outletTemperatureValueLabel.getText() + "\t";
    newData += outletConcValueLabel.getText() + "\n";
    outputTextArea.append(newData);
  }

  @Action
  public void saveDataSet() throws IOException {
    // Open a file chooser
    JFrame mainFrame = StaticReactorsApp.getApplication().getMainFrame();
    JFileChooser fc = new JFileChooser();
    int returnVal = fc.showSaveDialog(mainFrame);
    if (returnVal == JFileChooser.APPROVE_OPTION) {
      // the user selected a file
      File dataFile = fc.getSelectedFile();
      // open the file
      FileWriter outputStream = new FileWriter(dataFile);
      // write the data in the outputTextArea to the file
      outputTextArea.write(outputStream);
      // close the file
      outputStream.close();
    } else {
      // the user canceled
      return;
    }
  }

  public void enableAddExpt(boolean enable) {
    // only need to act if the button is not in the desired state
    if (enable != recordDataButton.isEnabled()) {
      // set the button to the desired state
      recordDataButton.setEnabled(enable);
    }
  }

  public void enableStartExpt(boolean enable) {
    // only need to act if the button is not in the desired state
    if (enable != startExptButton.isEnabled()) {
      // set the button to the desired state
      startExptButton.setEnabled(enable);
    }

  }

  public void setOutletUnsteady() {
    outletConcLabel.setText(unsteady1);
    outletConcValueLabel.setText(space);
    outletTemperatureLabel.setText(unsteady2);
    outletTemperatureValueLabel.setText(space);
  }
  // Variables declaration - do not modify//GEN-BEGIN:variables
  private javax.swing.JButton clearButton;
  private javax.swing.JFormattedTextField concAFormattedTextField;
  private javax.swing.JLabel concALabel;
  private javax.swing.JFormattedTextField concBFormattedTextField;
  private javax.swing.JLabel concBLabel;
  private javax.swing.JFormattedTextField concYFormattedTextField;
  private javax.swing.JLabel concYLabel;
  private javax.swing.JFormattedTextField concZFormattedTextField;
  private javax.swing.JLabel concZLabel;
  private javax.swing.JLabel concentrationLabel;
  private javax.swing.JLabel dataSetLabel;
  private javax.swing.JLabel diagramLabel;
  private javax.swing.JFormattedTextField flowRateFormattedTextField;
  private javax.swing.JLabel flowRateLabel;
  private javax.swing.JPanel inletSettingsPanel;
  private javax.swing.JLabel inletTemperatureLabel;
  private javax.swing.JScrollPane jScrollPane1;
  private javax.swing.JPanel mainPanel;
  private javax.swing.JMenuBar menuBar;
  private javax.swing.JLabel outletConcLabel;
  private javax.swing.JLabel outletConcValueLabel;
  private javax.swing.JPanel outletPanel;
  private javax.swing.JLabel outletTemperatureLabel;
  private javax.swing.JLabel outletTemperatureValueLabel;
  private javax.swing.JTextArea outputTextArea;
  private javax.swing.JButton recordDataButton;
  private javax.swing.JButton saveButton;
  private javax.swing.JMenuItem saveMenuItem;
  private javax.swing.JButton startExptButton;
  private javax.swing.JFormattedTextField temperatureFormattedTextField;
  private javax.swing.JMenuItem userGuideMenuItem;
  // End of variables declaration//GEN-END:variables
  private JDialog aboutBox, userGuide;
  private double inletTemperature, inletFlowRate;
  private double inletConcA, inletConcB, inletConcY, inletConcZ;
  private double vReactor, k0, actE, deltaH, rho, heatCapacity, gasConst;
  private double outletConcY, outletTemperature;
  private double concUncertainty, temperatureUncertainty;
  private double minimumInletT, maximumInletT;
  private double minimumInletFlow, maximumInletFlow;
  private double minimumInletConcA, maximumInletConcA;
  private double minimumInletConcB, maximumInletConcB;
  private double minimumInletConcY, maximumInletConcY;
  private double minimumInletConcZ, maximumInletConcZ;
  private int alpha, beta;
  private String unsteady1 = "The system has not yet";
  private String unsteady2 = "reached steady-state.";
  private String steady1 = "Concentration of Y (M)";
  private String steady2 = "Temperature (K)";
  private String calcFailed1 = "Something went wrong";
  private String calcFailed2 = "in the simulator.";
  private String calcFailed3 = "Try again with";
  private String calcFailed4 = "different inlet settings.";
  private String space = " ";
  private String header;
  private String reactorType;
  private double defaultInletTemperature, defaultInletFlowRate;
  private double defaultInletConcA, defaultInletConcB;
  private double defaultInletConcY, defaultInletConcZ;
}

